//////////////////////////////////////////////////////////////////////
//TargetD64 - C64 archive related conversion tool and emulator frontend
//////////////////////////////////////////////////////////////////////
//Copyright (C) 1998, 1999  Karlheinz Langguth klangguth@netscape.net
//
//This program is free software; you can redistribute it and/or
//modify it under the terms of the GNU General Public License
//as published by the Free Software Foundation; either version 2
//of the License, or (at your option) any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program; if not, write to the Free Software
//Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////
//COMPILE SWITCHES
// _MSC_VER
// indicates MS compiler
// _DEBUG
// for debug version which activates ASSERT and TRACE

#ifdef _MSC_VER
#pragma warning(disable:4786) //identifier truncation warning
#endif

#include <stdio.h>
#include <errno.h>
#ifdef _MSC_VER
#include <windows.h>
#include <tchar.h>
#else
#include <pwd.h>
#include <unistd.h>
#endif
#include <sys/types.h>
#include <sys/stat.h>

#include <algorithm>
#include <string>
#include <fstream>
#include <vector>
#include <map>

using namespace std;

#include "Profile.h"
#include "Tracing.h"


CFProfile* CFProfile::s_pAppProfile = NULL;


CFProfile::CFProfile(const string& appName, const string& profileDir /* = "" */)
{
	string dir = profileDir;

	if (dir.empty())
	{
		//profile resides in applications directory for Win32 - will be 
		//                in homedirectory of user for Linux
#ifdef _MSC_VER
		TCHAR home[_MAX_PATH] = { '\0' };

		if (::GetModuleFileName(NULL, home, _MAX_PATH) != 0)
		{
			//determine applications residence directory
			LPTSTR pDir = _tcsrchr(home, _T('\\'));
			if (pDir != NULL)
				*pDir = '\0';
		}
		else
		{
			home[0] = _T('.');
			home[1] = 0;
		}
		dir = home;
#else
		char *home = NULL;

		home = getenv("HOME");
		if (home == NULL)
		{
			struct passwd *pwd;

			pwd = getpwuid(getuid()); //check for passwd entry
			if ((pwd != NULL) && (pwd->pw_dir != NULL))
			{
				home = pwd->pw_dir;
			}
			else
				home = ".";
		}
		dir = home;
#endif
	}

#ifdef _MSC_VER
	m_profileName = dir + "\\" + appName + ".ini";
#else
	m_profileName = dir + "/." + appName + "rc";
#endif

	struct stat buf;
	if (::stat(m_profileName.c_str(), &buf) != 0)
	{
		if (errno == ENOENT)
		{
			//provide profile
			ofstream out(m_profileName.c_str(), ios::out);
		}
	}
}


//overreads white spaces
//R: one character after whitespaces
static char SkipWhiteSpaces(istream& in)
{
	char ch;
	do
	{
		ch = (char)in.get();
	}
	while (((ch == ' ') || (ch == '\t')) && in);
	//we have overread whitespaces now pushback character read too much
	return ch;
}


//remove leading and trailing blanks
//P1 IO: string to trim
static void Trim(string& a)
{
	string::size_type first = a.find_first_not_of(" \t");
	string::size_type last = a.find_last_not_of(" \t");
	if (last != string::npos)
		a = a.substr(0, last + 1);
	if (first != string::npos)
		a = a.substr(first, a.size() - first);
}


void CFProfile::ParseProfile(void) throw (CFProfileException)
{
	TRBEGIN("CFProfile::ParseProfile");

	m_sectionEntry.clear(); //new parse
	ifstream in(m_profileName.c_str(), ios::in); //open as text file!
	if (!in)
		throw CFProfileException();
	unsigned int lineCount = 0;
	string section;
	string entry;
	string value;
	while (in)
	{
		string line;
		char ch;
		char firstValidChar = ::SkipWhiteSpaces(in);
		if (in.eof())
			TRreturn;
		switch (firstValidChar)
		{
		case '\n':
			lineCount++;
			break;
		case ';':
			//ignore comments
			(void)getline(in, line);
			lineCount++;
			break;
		case '[':
			//found section
			section.resize(0);
			ch = ::SkipWhiteSpaces(in);
			//eval section name
			while ((ch != ']') && (ch != '\n') && in)
			{
				section += ch;
				ch = (char)in.get();
			}
			//remove leading and trailing blanks and or tabs
			::Trim(section);
			(void)m_sectionEntry.insert(make_pair(section,
				CFEntryLine(SECTIONMARKER, "", lineCount)));
			TRACE(<< "Entry insertion: " << section << "," << SECTIONMARKER << "," << "" << "," << lineCount);
			if (ch != ']')
			{
				//closing braket missing - assume section
				lineCount++;
			}
			else
			{
				//ignore rest of line
				(void)getline(in, line);
				lineCount++;
			}
			break;
		default:
			//if a section is already defined we have an entry
			if (!section.empty())
			{
				entry.resize(0);
				ch = firstValidChar;
				while ((ch != '=') && (ch != '\n') && in)
				{
					entry += ch;
					ch = (char)in.get();
				}
				::Trim(entry);
				if ((ch == '\n') || in.eof())
				{
					//we have no = sign - empty entry
					//mind that we artifically construct "EMPTYENTRY = <line>"
					(void)m_sectionEntry.insert(make_pair(section,
						CFEntryLine(EMPTYENTRY, entry, lineCount)));
					TRACE(<< "Entry insertion: " << section << "," << EMPTYENTRY << "," << entry << "," << lineCount);
					lineCount++;
				}
				else
				{
					if (ch == '=')
					{
						//everything behind = is value (maybe empty)
						(void)getline(in, value);
						::Trim(value);
						(void)m_sectionEntry.insert(make_pair(section,
							CFEntryLine(entry, value, lineCount)));
						TRACE(<< "Entry insertion: " << section << "," << entry << "," << value << "," << lineCount);
						lineCount++;
					}
				}
			}
			else
			{
				//stuff not related to a section - ignore
				(void)getline(in, line);
				lineCount++;
			}
			break;
		}
		if (in.eof())
			TRreturn;
	}
	if (in.bad())
	{
		m_sectionEntry.clear();
		throw CFProfileException();
	}
	TRreturn;
}


bool CFProfile::ReadProfileValue(const string& section, const string& entry, vector<CFEntryLine *>& aEntryLine)
{
	aEntryLine.resize(0);
	pair<tSectionEntry::iterator, tSectionEntry::iterator> interval;
	//search for section
	interval = m_sectionEntry.equal_range(section);
	//search for entry
	for (tSectionEntry::iterator i = interval.first; i != interval.second; ++i)
	{
		if (i->second.m_entry == entry)
		{
			aEntryLine.push_back(&(i->second));
		}
	}
	if (aEntryLine.empty())
		return false;
	else
		return true;
}


bool CFProfile::ReadProfileValue(const string& section, const string& entry, CFEntryLine*& pEntryLine)
{
	pair<tSectionEntry::iterator, tSectionEntry::iterator> interval;
	//search for section
	interval = m_sectionEntry.equal_range(section);
	//search for entry
	for (tSectionEntry::iterator i = interval.first; i != interval.second; ++i)
	{
		if (i->second.m_entry == entry)
		{
			pEntryLine = &(i->second);
			return true;
		}
	}
	return false;
}


bool CFProfile::ReadProfileValue(const string& section, const string& entry, string& value) const
{
	CFEntryLine *pFound = NULL;
	if (!const_cast<CFProfile *>(this)->ReadProfileValue(section, entry, pFound))
		return false;
	value = pFound->m_value;
	return true;
}


bool CFProfile::ReadProfileValue(const string& section, const string& entry, bool& arg) const
{
	string value;
	if (ReadProfileValue(section, entry, value))
	{
		//make lower case
		for (string::size_type i = 0; i < value.size(); i++)
			value[i] = (char)tolower(value[i]);
		//only accept boolean values anythin else not found
		if ((value == "1") || (value == "on") || (value == "yes") || (value == "true"))
		{
			arg = true;
			return true;
		}
		if ((value == "0") || (value == "off") || (value == "no") || (value == "false"))
		{
			arg = false;
			return true;
		}
	}
	return false;
}


string CFProfile::PointerToString(CFEntryLine *e)
{
	return e->m_value;
}


bool CFProfile::ReadProfileValue(const string& section, const string& entry, vector<string>& aValueEntry) const
{
	aValueEntry.resize(0);
	vector <CFEntryLine *> aFoundEntry;
	if (!const_cast<CFProfile *>(this)->ReadProfileValue(section, entry, aFoundEntry))
		return false;
	(void)transform(aFoundEntry.begin(), aFoundEntry.end(), back_inserter(aValueEntry), PointerToString);
	return true;
}


void CFProfile::WriteProfileValue(const string& section, const string& entry, const string& value) throw (CFProfileException)
{
	TRBEGIN("CFProfile::WriteProfileValue");
	TRACE(<< "Params: " << section << "," << entry << "," << value);

	ifstream in;
	ofstream out;

	CFEntryLine* pFound = NULL;
	if (ReadProfileValue(section, entry, pFound))
	{
		TRACE(<< "Entry found: " << section << "," << entry << "," << pFound->m_value << "," << pFound->m_lineNo);
		//we have section and entry already in profile - this will actualize cache
		pFound->m_value = value;
		InitProfileChange(in, out);
		//copy original profile until entry reached
		CopyProfileLines(in, out, pFound->m_lineNo);
		//Now write the changed line
		out << pFound->m_entry << "=" << value <<endl;
		if (!out)
			throw CFProfileException();
		//overread original entry
		SkipProfileLines(in, 1);
		//copy rest
		CopyProfileLines(in, out, ALL);
		CommitProfileChange(in, out);
	}
	else
	{
		if (ReadProfileValue(section, SECTIONMARKER, pFound))
		{
			TRACE(<< "Section only found: " << section << "," << SECTIONMARKER << "," << pFound->m_value << "," << pFound->m_lineNo);
			//we have the section but not the entry in profile
			InitProfileChange(in, out);
			//copy original profile including section start
			CopyProfileLines(in, out, pFound->m_lineNo + 1);
			//Now write the new entry line
			out << entry << "=" << value <<endl;
			if (!out)
				throw CFProfileException();
			//copy rest
			CopyProfileLines(in, out, ALL);
			CommitProfileChange(in, out);
			//now also actualize the cache (linenumber shift for new entry)
			for (tSectionEntry::iterator i = m_sectionEntry.begin()
				; i != m_sectionEntry.end(); ++i)
			{
				if (i->second.m_lineNo > pFound->m_lineNo)
					i->second.m_lineNo++;
			}
			//also add new entry
			(void)m_sectionEntry.insert(make_pair(section,
				CFEntryLine(entry, value, pFound->m_lineNo)));
		}
		else
		{
			TRACE(<< "Complete new section and entry");
			InitProfileChange(in, out);
			//write out section and entry
			out << "[" << section << "]" <<endl;
			//Now write the new entry line
			out << entry << "=" << value <<endl <<endl;
			if (!out)
				throw CFProfileException();
			//copy original profile
			CopyProfileLines(in, out, ALL);
			CommitProfileChange(in, out);
			//now also actualize the cache (linenumber shift for new entry)
			for (tSectionEntry::iterator i = m_sectionEntry.begin()
				; i != m_sectionEntry.end(); ++i)
			{
				i->second.m_lineNo += 3;
			}
			//add new section
			(void)m_sectionEntry.insert(make_pair(section,
				CFEntryLine(SECTIONMARKER, "", 0)));
			//add new entry
			(void)m_sectionEntry.insert(make_pair(section,
				CFEntryLine(entry, value, 1)));
		}
	}
	TRreturn;
}


void CFProfile::InitProfileChange(ifstream& in, ofstream& out) const throw (CFProfileException)
{
	in.open(m_profileName.c_str(), ios::in); //open as text file!
	if (!in)
		throw CFProfileException();
	out.open((m_profileName + ".new").c_str(), ios::out);
	if (!out)
		throw CFProfileException();
}


void CFProfile::CommitProfileChange(ifstream& in, ofstream& out) const throw (CFProfileException)
{
	in.close();
	out.close();
#ifdef _MSC_VER
	(void)::remove(m_profileName.c_str());
#endif
	if (::rename((m_profileName + ".new").c_str(), m_profileName.c_str()))
	{
		throw CFProfileException();
	}
}


void CFProfile::CopyProfileLines(ifstream& in, ofstream& out, const unsigned int nLines) const throw (CFProfileException)
{
	if (nLines == ALL)
	{
		char c;
		while (in.get(c))
			(void)out.put(c);
		if (in.bad() || !out)
			throw CFProfileException();
	}
	else
	{
		for (unsigned int i = 0; i < nLines; i++)
		{
			string line;
			(void)getline(in, line);
			if (in.bad())
				throw CFProfileException();
			out << line <<endl;
			if (!out)
				throw CFProfileException();
		}
	}
}


void CFProfile::SkipProfileLines(ifstream& in, const unsigned int nLines) const throw (CFProfileException)
{
	for (unsigned int i = 0; i < nLines; i++)
	{
		string line;
		(void)getline(in, line);
		if (in.bad())
			throw CFProfileException();
	}
}
