//////////////////////////////////////////////////////////////////////
//TargetD64 - C64 archive related conversion tool and emulator frontend
//////////////////////////////////////////////////////////////////////
//Copyright (C) 1998, 1999  Karlheinz Langguth klangguth@netscape.net
//
//This program is free software; you can redistribute it and/or
//modify it under the terms of the GNU General Public License
//as published by the Free Software Foundation; either version 2
//of the License, or (at your option) any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program; if not, write to the Free Software
//Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////
//COMPILE SWITCHES
// _MSC_VER
// indicates MS compiler
// _DEBUG
// for debug version which activates ASSERT and TRACE
// TD64_MODIFIED
// marks changes in foreign sources to fit into TargetD64
// must be set everywhere because also used for header files
// NO_GUI
// Compile TD64 as a pure console application (as it has been intended)
// Well, Win32 user force you to implement a GUI ;-)

#ifdef _MSC_VER
#pragma warning(disable:4786) //identifier truncation warning
#endif

#include <signal.h>

#ifdef _MSC_VER
#include <windows.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>

#include <algorithm>
#include <map>
#include <vector>
#include <string>
#include <iostream>
#include <fstream>
#include <strstream>

using namespace std;

#include "Tracing.h"
#include "Exception.h"
#include "Options.h"
#include "Profile.h"
#include "Application.h"
#include "TargetD64.h"
#include "Image.h"
#include "DiskImage.h"
#include "Archive.h"
#include "HostArchive.h"
#ifndef NO_GUI
#include "GUI.h"
#endif

static const string version =
VERSION
"\nan universal C64 archive related converter tool and emulator frontend\n\
Copyright (C) 1998, 1999 Karlheinz Langguth\n\
Unlynx, C64zip code copyright (C) 1999 The VICE Team\n\
Unzip, Gunzip code copyright (C) 1998 Mark Adler, Jean-loup Gailly, Gi. Vollant\n\
Unlha code Copyright (C) 1992 Y. Tagawa, Masaru Oki\n\n"
"This program is free software; you can redistribute it and/or\n\
modify it under the terms of the GNU General Public License\n\
as published by the Free Software Foundation; either version 2\n\
of the License, or (at your option) any later version.\n\
This program is distributed in the hope that it will be useful,\n\
but WITHOUT ANY WARRANTY; without even the implied warranty of\n\
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n\
GNU General Public License for more details.\n\n"
HOMEPAGE
MAIL
"Use -h for command syntax.\n";

static const string help =
"\
NAME\n\
	targetd64 - an universal C64 archive related converter tool\n\
	            and emulator frontend\n\
\n\
SYNOPSIS\n\
	targetd64 [-h] [-v] [-n] [-d] [-t tracefile] [-k]\n\
	          [-c targetdir] [file ...]\n\
\n\
DESCRIPTION\n\
	TargetD64 processes the files given at command line resulting in one\n\
	or more D64 image(s) containing unresolvable C64 files. Those D64\n\
	images are put into a temporary directory. Dearchiving is executed\n\
	recursively. Supported archive formats are zip, gz, lha, lnx, [1-4]!,\n\
	d64, x64, t64, p00. If a file can not be matched to one of these\n\
	formats it is treated as a raw C64 (PRG) file. Resulting D64 images are\n\
	named target001.d64 to targetXXX.d64. An emulator is launched with\n\
	target001.d64. Repository information (like D64 directory) is passed to\n\
	the emulator. After exiting the emulator all traces of the processing\n\
	(even the D64 images for the emulator) are deleted.\n\
\n\
OPTIONS\n\
	-h\n\
	prints out this help text.\n\
\n\
	-v\n\
	writes out the version of targetd64.\n\
\n\
	-n\n\
	preserves the directory tree built while processing after program\n\
	termination. The directory tree is located in the temporary directory\n\
	(see environment variable TMPDIR).\n\
\n\
	-d\n\
	stands for debug. TargetD64 comes heavily instrumented for the beta\n\
	releases. This option will provide a runtime trace on stderr (redirect\n\
	the output of file handle 2 to a file because it will be a lot of\n\
	information). This is mainly for me as the developer. On the other hand\n\
	you see every call of external applications. So it could be useful for\n\
	you.\n\
\n\
	-t tracefile\n\
	is the same than -d except that the trace output is dumped into a file\n\
	named tracefile instead stderr. This is especially introduced for poor\n\
	command interpreters that are not able to redirect stderr. Use -t\n\
	stdout to dump the trace output into stdout.\n\
\n"
"\
	-c targetdir\n\
	stands for conversion only. With targetdir you specify a directory for\n\
	the resulting D64 images. No emulator will be launched. After\n\
	execution you will find the resulting D64 images in targetdir.\n\
\n\
	-k\n\
	stands for keep archive filename. Parts of the filename of the\n\
	processed archive are used to name the resulting D64 image. This should\n\
	make it easier to relate the D64 image to its source when specified\n\
	multiple files for processing.\n\
\n\
FILES\n"
#ifdef _MSC_VER
"	<targetd64 dir>\\targetd64.ini\n\
		profile file in the same directory as targetd64.exe.\n"
#else
"	$HOME/.targetd64rc\n\
		profile file in user's home directory.\n"
#endif	
"\n	Mind that command line options have priority over environment\n\
	variables which have priority over profile settings.\n\
\n\
ENVIRONMENT\n"
#ifdef _MSC_VER
"\
	TMPDIR, TMP, TEMP\n\
	The value of TMPDIR has priority over TMP has priority over TEMP.\n\
	If none of these are set c:\\temp is set as the temporary directory.\n\
	TargetD64 creates the top of its temporary directory tree in this\n\
	directory.\n\
\n\
	TD64_EMU_CMD\n\
	The command to launch the emulator. This command is passed to the shell\n\
	so you may use the complete shell syntax in general. Use ; to separate\n\
	multiple commands. The following placeholders are replaced by their\n\
	actual values at runtime:\n\
	%s1 %s2 %s3 %s4 %l1 %l2 %l3 %l4 %d1 %d2 %d3 %d4\n\
	%s1 is replaced by the short (8.3 as DOS executables need it) pathname\n\
	of the directory where the D64 images reside. There is NO trailing \\.\n\
	%s2 is replaced by the short (8.3 as DOS executables need it) pathname\n\
	of the first emulator image for the emulator .\n\
	%s3 is replaced by the basename of %s2 which means stripped off\n\
	directory path.\n\
	%s4 is replaced by the short pathname (8.3 as DOS executables need it)\n\
	of temporary directory (see TMPDIR).\n\
	%l<n> is replaced by the long pathname version of %s<n> as Win32\n\
	applications can handle.\n\
	%d<n> is replaced by the drive info of %s<n> (e.g. 'c:').\n\
\n"
"\
	TD64_GENERIC_EXTx, TD64_GENERIC_CMDx\n\
	generic archive interface to hook in unknown archive formats.\n\
	To handle a new archive format you have to define the pair of\n\
	environment variables stated above where x stands for a number.\n\
	Mind that x must be a continuation starting with 1 (NO GAPS!).\n\
	TD64_GENERIC_EXTx sets the extension (must have preceding dot).\n\
	TD64_GENERIC_CMDx sets the command to extract the archive.\n\
	For TD64_GENERIC_CMDx the following placeholders are replaced by\n\
	their actual values at runtime: %s1 %s2 %s3 %l1 %l2 %l3 %d1 %d2 %d3\n\
	%s1 is replaced by the short (8.3) pathname of the directory where the\n\
	files are put to while extracting the archive. There is NO trailing \\.\n\
	%s2 is replaced by the short (8.3) pathname of the archive to extract.\n\
	%s3 is replaced by the short (8.3) pathname of the archive to extract\n\
	without dirname and extension.\n\
	%l<n> and %d<n> are replaced accordant to TD64_EMU_CMD.\n"
#else
"\
	TMPDIR\n\
	TargetD64 creates the top of its temporary directory tree in this\n\
	directory. If it is not set the default /tmp is used.\n\
\n\
	TD64_EMU_CMD\n\
	The command to launch the emulator. This command is passed to the shell\n\
	so you may use the complete shell syntax in general. The following\n\
	placeholders are replaced by their actual values: %s1 %s2 %s3 %s4\n\
	%s1 is replaced by the pathname of the directory where the D64 images\n\
	reside. There is NO trailing /.\n\
	%s2 is replaced by the pathname of the first emulator image for the\n\
	emulator.\n\
	%s3 is replaced by the basename of %s2 which means stripped off\n\
	directory path.\n\
	%s4 is replaced by the temporary directory (see TMPDIR).\n\
\n\
	TD64_GENERIC_EXTx, TD64_GENERIC_CMDx\n\
	generic archive interface to hook in unknown archive formats.\n\
	To handle a new archive format you have to define the pair of\n\
	environment variables stated above where x stands for a number.\n\
	Mind that x must be a continuation starting with 1 (NO GAPS!).\n\
	TD64_GENERIC_EXTx sets the extension (must have preceding dot).\n\
	TD64_GENERIC_CMDx sets the command to extract the archive.\n\
	For TD64_GENERIC_CMDx the following placeholders are replaced by\n\
	their actual values at runtime: %s1 %s2 %s3\n\
	%s1 is replaced by the pathname of the directory where the files are put\n\
	to while extracting the archive. There is NO trailing /.\n\
	%s2 is replaced by the pathname of the archive to extract.\n\
	%s3 is replaced by the pathname of the archive to extract without dirname\n\
	and extension.\n"
#endif
"\n\
	TD64_DEFAULT_OPTIONS\n\
	To make TargetD64 completely configurable by environment variables any\n\
	option(s) that have to be handled with each call of TargetD64 can be\n\
	stated in TD64_DEFAULT_OPTIONS. Those are always prefixed to the\n\
	options given at command line.\n\
\n\
SEE ALSO\n\
\n\
DIAGNOSTICS\n\
	Successful execution is indicated by exit code 0. Any other indicates\n\
	error.\n\
\n\
BUGS\n\
\n\
AUTHOR\n\
"
"	"HOMEPAGE
"	"MAIL;


const string appName("targetd64"); //is used as part of profile name

//my special application - good luck
class CFTargetD64 : public CFApplication
{
public:
	CFTargetD64(int argc, char *argv[] = NULL, char *envp[] = NULL);

public:
	virtual int main(void) const;
private:
	//read settings from profile, command line, environment
	void ReadSettings(void);

private:
	static void SignalHandler(int sigNo) throw (CFException);
	static void SetSignalHandler(void (*handler)(int));
	static void HandleOptionVersion(const string&);
	static void HandleOptionHelp(const string&);

private:
	vector<string> m_files; //all files to be processed
	string m_emuImageDirPath;

private:
	static string m_emulatorCommand;
	//the mother of all archives ever processed in this call
	static CFHostArchive m_rootArchive;
	static bool m_bCleanup; //false means do not cleanup
};


//the commands to start up the emulator
//%s1 for directory where images lay with trailing slash (backslash)
//%s2 for path of first image to start with
//%s3 for basename of first image
//%s4 for tmp directory
#ifdef _MSC_VER
//can not set source dir for ccs64, so always place images in a permanent dir
//1. make a permanent directory to place the images always into
//2. delete anything within this directory so move will never ask about overwrite (hang up!)
//3. move the images from the temp directory to the permanent directory
//4. copy back only the first emu image from permanent to tmp dir to be found by emulator
//5. call the emulator with the first image in the permanent directory
//6. delete the images from the permanent directory
//ATTENTION: FOR WIN95 USE "filename" IN SHELL COMMAND AS + CHARACTER MAY BE
//           INTERPRETED BY THE SHELL WITHOUT QUOTES
string CFTargetD64::m_emulatorCommand = "mkdir %l4\\images; del %l4\\images\\*.d64; move %l1\\*.d64 %l4\\images; copy \"%l4\\images\\%l3\" %l1; call ccs64_95 %s2,0; del %l4\\images\\*.d64";
#else
string CFTargetD64::m_emulatorCommand = "cd '%s1' && x64 -8 '%s2'";
#endif

CFTargetD64::CFTargetD64(int argc, char *argv[], char *envp[])
	: CFApplication(appName, argc, argv, envp)
{
	(void)m_opt.RegisterOption(OPT_HELP.first, OPT_HELP.second, HandleOptionHelp);
	(void)m_opt.RegisterOption(OPT_TRACE.first, OPT_TRACE.second);
	(void)m_opt.RegisterOption(OPT_TRACE_FILE.first, OPT_TRACE_FILE.second);
	(void)m_opt.RegisterOption(OPT_NO_CLEANUP.first, OPT_NO_CLEANUP.second);
	(void)m_opt.RegisterOption(OPT_CONVERT_ONLY.first, OPT_CONVERT_ONLY.second);
	(void)m_opt.RegisterOption(OPT_VERSION.first, OPT_VERSION.second, HandleOptionVersion);
	(void)m_opt.RegisterOption(OPT_KEEP_FILENAME.first, OPT_KEEP_FILENAME.second);
	(void)m_opt.RegisterOption(OPT_FILENAME.first, OPT_FILENAME.second);

	char *env;
	//read default options from environment variable and parse them as well
	if ((env = ::getenv("TD64_DEFAULT_OPTIONS")) != NULL)
	{
		istrstream optionStream(env);
		vector<string> options;
		string singleOption;
		//extract the single options of the command line
		while (optionStream >> singleOption)
		{
			 options.push_back(singleOption);
		}
		//build up argv array
		if (!options.empty())
		{
			//alloc array of char pointers
			char **argvEnv = new char* [options.size() + 1];
			ASSERT(argvEnv != NULL);
			//offset 1, because index 0 is reserved for prog name
			for (vector<string>::size_type i = 1; i < options.size() + 1; i++)
			{
				//link in the single options
				argvEnv[i] = const_cast<char *>(options[i - 1].c_str());
			}
			m_opt.ParseCommandLine((int)options.size() + 1, argvEnv);
			delete [] argvEnv;
		}
	}
	m_opt.ParseCommandLine(argc, argv);

	ReadSettings();
}


void CFTargetD64::SetSignalHandler(void (*handler)(int))
{
#ifndef _MSC_VER
    (void)::signal(SIGBUS, handler);
    (void)::signal(SIGHUP, handler);
    (void)::signal(SIGQUIT, handler);
#endif
    (void)::signal(SIGILL, handler);
    (void)::signal(SIGINT, handler);
    (void)::signal(SIGABRT, handler);
    (void)::signal(SIGFPE, handler);
    (void)::signal(SIGSEGV, handler);
    (void)::signal(SIGTERM, handler);
}


void CFTargetD64::SignalHandler(int sigNo) throw (CFException)
{
	cerr << endl
	<< "----------------------------------------------------" <<endl
	<< "--------Signal occured - program terminated---------" <<endl
	<< "----------------------------------------------------" <<endl;
	cerr << "Signal ID    : " << sigNo <<endl;
	SetSignalHandler(SIG_IGN);

	if (m_bCleanup) //only cleanup if wanted (debugging)
	{
		m_rootArchive.CleanUpRootArchiveFilesystem();
	}
	exit(1);
}


void CFTargetD64::HandleOptionHelp(const string&)
{
	cout << help;
	exit(0);
}


void CFTargetD64::HandleOptionVersion(const string&)
{
	//print out the version string
	cout << version;
	exit(0);
}


//Be careful with constructors of static objects:
//the constructor of the static topmost archive intitializes the
//static members of CFArchive. It appeared that these static
//members are initialized after the constructor call again
//by M$-compiler effecting in empty static members (which have
//been previously initialized). This is probably an error.

//the mother of the archive hirarchy
CFHostArchive CFTargetD64::m_rootArchive;
bool CFTargetD64::m_bCleanup = true; //process cleanup


//return the latest found filename P2 matching the wildcard
//filename P1. after R has returned false for the first time
//next call will start NEW search!
//Filter out unwanted directory entries. those are not returned.
//ATTENTION: always call this routine until R is false
//otherwise it will not recognize change of pattern P1!!!!
//P1 I: wildcard filename to find
//P2 O: found long filename
//R: false means no more matches found - do not eval
//   P2 except for first call (always eval P2 after first call)
static bool SearchFileFromWildcard(const string& path
							  ,string &filenameFound)
{
#ifdef _MSC_VER
	static char *ignore;
	static char name[_MAX_PATH + 1] = { '\0' };
	static HANDLE handle = INVALID_HANDLE_VALUE;
	WIN32_FIND_DATA data;

	//check if new call for fifferent pattern has happened
	if (handle == INVALID_HANDLE_VALUE)
	{
		if (GetFullPathName(path.c_str(), _MAX_PATH, name, &ignore))
		{
			//we use FindFirstFile to get long filename!
			handle = FindFirstFile(name, &data);
			if (handle == INVALID_HANDLE_VALUE)
			{
				filenameFound = name;
				return false;
			}
		}
		else
		{
			filenameFound = path;
			return false;
		}
	}
	else
	{
		if (!FindNextFile(handle, &data))
		{
			(void)FindClose(handle);
			handle = INVALID_HANDLE_VALUE;
			filenameFound = path;
			return false;
		}
	}
	//at this point we have a valid filled data structure
	//now if we have revealed a directory dump it
	while (data.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)
	{
		if (!FindNextFile(handle, &data))
		{
			(void)FindClose(handle);
			handle = INVALID_HANDLE_VALUE;
			filenameFound = path;
			return false;
		}
	}

	char drive[_MAX_DRIVE + 1]; //nice stack size
	char dirname[_MAX_PATH + 1];
	//get the directory path of filename
	_splitpath(name, drive, dirname, NULL, NULL);
	//set everything together
	filenameFound = string(drive) + string(dirname)
		+ string(data.cFileName);
	return true;
#else
	//on Linux wildcard expansion is done by shell - do nothing
	filenameFound = path;	
	return false;
#endif
}



//as name tells
//we only have to consider UNIX like slashes as only those are appended
//while processing
//P1 IO: directory name with potential trailing slashes which are removed
inline static void StripOffTrailingSlashesOfDirectory(string& dir)
{
	while ((dir.size() > 1) && (dir[dir.size() - 1] == '/'))
		dir.resize(dir.size() - 1);
}


void CFTargetD64::ReadSettings(void)
{
	//evaluate trace options BEFORE TRBEGIN to not screw up indentation
	bool flag;
	string value;
	if (m_profile.ReadProfileValue(PRO_OPT_TRACE.first, PRO_OPT_TRACE.second, flag)
		&& flag)
	{
		TRACEON;
	}
	if (m_profile.ReadProfileValue(PRO_OPT_TRACE_FILENAME.first, PRO_OPT_TRACE_FILENAME.second, value))
	{
		TRACEON;
		TRACEATTACHFILE(value);
	}
	if (m_opt.QueryOption(OPT_TRACE.first, value))
	{
		TRACEON;
	}
	if (m_opt.QueryOption(OPT_TRACE_FILE.first, value))
	{
		TRACEON;
		TRACEATTACHFILE(value);
	}

	TRBEGIN("CFTargetD64::ReadSettings");

	//-----------------------------------------------------------
	//Profile settings
	//-----------------------------------------------------------
	if (m_profile.ReadProfileValue(PRO_OPT_NOCLEANUP.first, PRO_OPT_NOCLEANUP.second, flag)
		&& flag)
	{
		m_bCleanup = false;
	}
	(void)m_profile.ReadProfileValue(PRO_OPT_CONVERTONLY.first, PRO_OPT_CONVERTONLY.second, m_emuImageDirPath);
	(void)m_profile.ReadProfileValue(PRO_EMULATOR_CMD.first, PRO_EMULATOR_CMD.second, CFTargetD64::m_emulatorCommand);

	//-----------------------------------------------------------
	//Command line options
	//-----------------------------------------------------------
	//no cleanup
	m_bCleanup &= !m_opt.QueryOption(OPT_NO_CLEANUP.first, value);
	//conversion only
	(void)m_opt.QueryOption(OPT_CONVERT_ONLY.first, m_emuImageDirPath);
	//filenames to be processed
	(void)m_opt.QueryOption(OP_STD_ARG, m_files);

	//-----------------------------------------------------------
	//Environment variables
	//-----------------------------------------------------------
	char *env;
	if ((env = ::getenv("TD64_EMU_CMD")) != NULL)
	{
		TRACE(<< "Emulator launch command set to: " << env);
		CFTargetD64::m_emulatorCommand = env;
	}

	TRreturn;
}


int CFTargetD64::main(void) const
{
	TRBEGIN("CFTargetD64::main");

	int exitCode = 0; //to signal errors to caller
	//Check the environment for emulator command
	//if no file on the command line given show error
	if (m_files.empty())
	{
		cerr << "Missing filename error." << endl;
		cerr << "Use -h for command syntax." <<endl;
#ifndef NO_GUI
		StartGUI(); //for now just give warning
#endif
		exit(1);
	}
		
	try {
	//CALL THIS METHOD BEFORE PROCESSING AN ARCHIVE
	m_rootArchive.InitRootArchiveAndEnvironment("target", m_emuImageDirPath);

	//Install handler for signals to clean up filesystem
	SetSignalHandler(SignalHandler);

	int nInputFiles = 0; //counter for input files

	//for each archive given at command line
	for (vector<string>::size_type i = 0; i < m_files.size(); i++)
	{
		TRACE(<< "Archive[" << i <<"] is " << m_files[i]);
		//create a temporary archive and copy the file to the working
		//directory. CopyThisArchiveToDirectory is virtual. For Multi-
		//file archives (e.g. C64Zip) it copies ALL files at once

		//in case we are using drag and drop on Windows
		//a short filename (8.3) is passed to the command line
		//to fix this try to get the long name instead (sigh)
		string filename;

		//search for wildcards and iterate over matches
		//for Linux the input filename is just returned as
		//wildcard expansion is done by shell
		bool firstCall = ::SearchFileFromWildcard(m_files[i], filename);

		bool ret = true; //one filename always returned!
		while (ret)
		{
#ifdef _MSC_VER
			//meaningless for Linux
			TRACE(<< "Short filename >" << m_files[i] << "< leads to long filename >"
				<< filename << '<');
#endif

#ifdef NOCOPY
			//TODO: this NOCOPY stuff will fail for multipart
			//archives when only ONE part is given as command line
			//argument - before the missing parts were copied!!!
			struct stat buf;
			if (::stat(filename.c_str(), &buf) != 0)
			{
				//OK this is an unusual exception ID
				//it is used for compatibility, before there has
				//been a filecopy which opened the file (not stat)
				exc = CFException(CFException::FILE_OPEN_FAILED,
					__LINE__, __FILE__, filename, "reading");
				throw exc;
			}
			else
			{
				//only regular files are to be processed
				if (!S_ISREG(buf.st_mode))
				{
					exc = CFException(CFException::UNEXPECTED_FILE_TYPE,
					__LINE__, __FILE__, filename, CFException::IntToString(buf.st_mode));
					throw exc;
				}
				//calculate the blocksize of the file
				unsigned int blockSize =
					((unsigned int)buf.st_size +  BLOCKSIZE_NETTO - 1) / BLOCKSIZE_NETTO;
				CFArchive *newEntry = m_rootArchive.CreateSubClassObject(filename, blockSize);
				m_rootArchive.AddNewArchiveToChildren(newEntry);
				nInputFiles++; //one file more to process
			}
#else
			CFArchive *tmp = m_rootArchive.CreateSubClassObject(filename, 0);
			try {
			nInputFiles++; //assume correct copying - if it fails correction later
			tmp->CopyThisArchiveToDirectory(m_rootArchive.GetDirname());
			}
			catch (CFException& actExc)
			{
				actExc.WriteOutExceptionWithWarningHeader(cerr);
				nInputFiles--; //in case of error no input file
			}
			delete tmp;
#endif

			if (!firstCall) //if nothing more to find break
				break;

			ret = ::SearchFileFromWildcard(m_files[i], filename);
		}
	}
	
	//if nothing to be processed do nothing
	//this will avoid unnecessary exeption messages
	if (nInputFiles == 0)
	{
		exc = CFException(CFException::APPLICATION_ERROR,
			__LINE__, __FILE__, "", "");
		throw exc;
	}
		
	//start the iterative processing
	m_rootArchive.ProcessRootArchive();

	ASSERT(!CFArchive::GetFirstEmuImageArchive().GetPathname().empty());

	//if emu image dir given no emulator call
	if (m_emuImageDirPath.empty())
	{
		const CFArchive& firstArchive = CFArchive::GetFirstEmuImageArchive();
		//%s1 is directory where emu images reside (without trailing /)
		//%s2 is pathname of first emu image
		//%s3 is basename of first emu image
		//%s4 is tmp directory (without trailing /)
		string emuImageDir = CFArchive::GetEmuImageDir();
		StripOffTrailingSlashesOfDirectory(emuImageDir);
		string command = CFArchive::ReplacePlaceHoldersByStrings(m_emulatorCommand
				, emuImageDir
				, firstArchive.GetPathname()
				//because of Win32 long and short filename nonsense we
				//have to determine the basename in the replace method
				//as it will be different for %sX and %lX
				, "basename " + firstArchive.GetPathname()
				, CFArchive::GetTmpDir());
		TRACE(<< "Execute command: " << command);

		//just use redirected output for system call
		int ret = CFArchive::SystemRedirectOutput(command.c_str());
		if (ret != 0)
		{
			exc = CFException(CFException::EMULATOR_CALL_FAILED,
				__LINE__, __FILE__, command, CFException::IntToString(ret));
			exc.WriteOutExceptionWithWarningHeader(cerr);
		}
		TRACE(<< "Returncode system call for emulator start: " << ret);
	}

	} //END TRY
	catch (CFException& actExc)
	{
		exitCode = 1; //we have encountered a fatal error - set exit code
		actExc.WriteOutExceptionWithErrorHeader(cerr);
	}
	
	//ignore signals while cleaning up
	SetSignalHandler(SIG_IGN);
	try {

	if (m_bCleanup) //only cleanup if wanted (debugging)
	{
		m_rootArchive.CleanUpRootArchiveFilesystem();
	}

	} //end try
	catch (CFException& actExc)
	{
		//cleanup raised exception
		actExc.WriteOutExceptionWithErrorHeader(cerr);
	}

	TRreturn exitCode;
}

//this is the main function of the application
//it passes the parameters to the application object
int main(int argc, char *argv[], char *envp[])
{
	CFTargetD64 myApp(argc, argv, envp);

	int exitCode = 0;
#ifndef NO_GUI
	exitCode = InitGUI();
#endif
	//call the application
	if (exitCode == 0)
		exitCode = myApp.main(); //will set exit code according processing state
	exit(exitCode);
	return 0;
}
