//////////////////////////////////////////////////////////////////////
//TargetD64 - C64 archive related conversion tool and emulator frontend
//////////////////////////////////////////////////////////////////////
//Copyright (C) 1998, 1999  Karlheinz Langguth klangguth@netscape.net
//
//This program is free software; you can redistribute it and/or
//modify it under the terms of the GNU General Public License
//as published by the Free Software Foundation; either version 2
//of the License, or (at your option) any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program; if not, write to the Free Software
//Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////
//COMPILE SWITCHES
// _MSC_VER
// indicates MS compiler
// _DEBUG
// for debug version which activates ASSERT and TRACE
// TD64_MODIFIED
// marks changes in foreign sources to fit into TargetD64
// must be set everywhere because also used for header files

#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <vector>
#include <string>
#include <iostream>
#include <fstream>
#include <strstream>
#ifdef _MSC_VER
#include <typeinfo.h>
#else
#include <typeinfo>
#endif

using namespace std;

#include "Exception.h"
#include "Image.h"
#include "TapeImage.h"
#include "Tracing.h"


CFTapeImage::CFTapeImage(void)
{
	ASSERT(sizeof(tTapeRecord) == 64);
	ASSERT(sizeof(tFileRecord) == 32);
	memset(&m_tapeRecord, sizeof(tTapeRecord), 0);
}


void CFTapeImage::Attach(const string& filename)
	throw (CFException)
{
	m_filename = filename;
	ASSERT(!m_inStream.rdbuf()->is_open()); //assure that the stream is closed
	m_inStream.open(m_filename.c_str(), ios::in | ios::binary);
	if (!m_inStream)
	{
		exc = CFException(CFException::FILE_OPEN_FAILED,
			__LINE__, __FILE__, m_filename, "reading");
		throw exc;
	}
	(void)m_inStream.read(reinterpret_cast<char *>(&m_tapeRecord)
		, sizeof(tTapeRecord));
	if ((size_t)(m_inStream.gcount()) != sizeof(tTapeRecord))
	{
		exc = CFException(CFException::TAPE_IMAGE_TOO_SMALL,
			__LINE__, __FILE__, m_filename, "");
		throw exc;
	}
	//a few consistency checks
	if (GetEntryCount() > GetMaxEntryCount())
	{
		exc = CFException(CFException::TAPE_IMAGE_NOT_ENOUGH_ENTRIES,
			__LINE__, __FILE__, m_filename, "");
		throw exc;
	}
	for (unsigned int i = 0; i < GetMaxEntryCount(); i++)
	{
		tFileRecord fileRecord;
		(void)m_inStream.read(reinterpret_cast<char *>(&fileRecord)
			, sizeof(tFileRecord));
		if ((size_t)m_inStream.gcount() != sizeof(tFileRecord))
		{
			exc = CFException(CFException::TAPE_IMAGE_TOO_SMALL,
				__LINE__, __FILE__, m_filename, "");
			throw exc;
		}
		//append file record only if occupied.
		if (fileRecord.entryType != FREE)
		{
			//correct the c64 type if necessary
			//T64 format is poorly defined and implemented by other tools!!!
			//i have encountered tapes which c64 filtype == 1 which means
			//normally *SEQ. Set it to PRG (like other conversion programs do)
			//if file not closed or any of the unused bits used map to PRG
			//Yes this is a hack.
			if (!(fileRecord.c64DiskFileType & 0x80)
				|| (fileRecord.c64DiskFileType & 0x38))
			{
				fileRecord.c64DiskFileType = 0x82; //PRG
			}
			m_fileRecords.push_back(fileRecord);
		}
		if (fileRecord.entryType > STREAM)
		{
			//limit the c64 filename - dirty hack but works
			fileRecord.c64Filename[sizeof(fileRecord.c64Filename) - 1] = '\0';
			exc = CFException(CFException::TAPE_IMAGE_UNKNOWN_FILE_TYPE,
				__LINE__, __FILE__, m_filename, fileRecord.c64Filename);
			throw exc;
		}
	}
	if (!m_inStream)
	{
		exc = CFException(CFException::FILE_OPERATION_FAILED,
			__LINE__, __FILE__, m_filename, "reading");
		throw exc;
	}
}


void CFTapeImage::ExtractFile(const tFileRecord& entry
	, const string& dirname
	, CFExtractedFileInfo& extractInfo
	, const string& filename /*= ""*/) const throw (CFException)
{
	ASSERT(!dirname.empty());

	//normally I would have used algo "find" but I could not convince
	//compiler of const character of this routine
	unsigned int nRecIdx; //index of ile record within tape for file to extract
	for (nRecIdx = 0; nRecIdx < m_fileRecords.size(); nRecIdx++)
	{
		//can not use operator == for M$
		if (memcmp(&entry, &(m_fileRecords[nRecIdx]), sizeof(entry)) == 0)
			break;
	}
	//entry must be from vector and nowhere else
	ASSERT(nRecIdx < m_fileRecords.size());

	//Get the CBM filename and limit it to 16 characters
	//there are is no trailing '\0' for CBM strings
	char tmp[sizeof(entry.c64Filename) + 1];
	strncpy(tmp, entry.c64Filename, sizeof(entry.c64Filename));
	//remove padded blanks (this is wrong for filenames that have
	//really trailing blanks in their names - no way to tell for T64).
	int i;
	for (i = sizeof(entry.c64Filename); i > 0; i--)
	{
		if (tmp[i - 1] != 0x20)
		{
			tmp[i] = '\0';
			break;
		}
	}
	if (i == 0) //take care of empty filename (16 blanks)
		tmp[0] = '\0';
	CFCbmFilename cbmFilename(tmp);

	//calc length of file
	int length = 0;
	unsigned int offset = GetOffset(m_fileRecords[nRecIdx]);
	if (offset < GetMaxEntryCount() * sizeof(tFileRecord) + sizeof(tTapeRecord))
	{
		exc = CFException(CFException::TAPE_FILE_OFFSET_OUT_OF_BOUND_ERROR,
			__LINE__, __FILE__, m_filename, cbmFilename);
		throw exc;
	}
	if (nRecIdx < m_fileRecords.size() - 1)
	{
		unsigned int nextOff = GetOffset(m_fileRecords[nRecIdx + 1]);
		if (nextOff > offset)
		{
			//length is offset difference of neighbor files
			length = (int)(nextOff -  offset);
		}
		else
		{
			//offsets not in row - corrupted tape image
			exc = CFException(CFException::TAPE_FILE_OFFSETS_ERROR,
				__LINE__, __FILE__, m_filename, cbmFilename);
			throw exc;
		}
	}
	//strip off trailing / if existing
	string fsDirname = 	dirname;
	if (fsDirname[fsDirname.size() - 1] == '/')
	{
		fsDirname.resize(fsDirname.size() - 1);
	}

	//check existance of directory
	struct stat buf;
	if (stat(fsDirname.c_str(), &buf) != 0)
	{
		exc = CFException(CFException::FILE_STAT_FAILED,
			__LINE__, __FILE__, fsDirname, "", errno);
		throw exc;
	}
	//only directory is allowed
	if (!S_ISDIR(buf.st_mode))
	{
		exc = CFException(CFException::UNEXPECTED_FILE_TYPE,
			__LINE__, __FILE__, fsDirname, CFException::IntToString(buf.st_mode));
		throw exc;
	}

	ofstream out;
	string fsFilename(fsDirname + "/" + filename);

	if (filename != "")
	{
		//if explicit filename is given use it in any case
		out.open(fsFilename.c_str(), ios::out | ios::binary);
	}
	else
	{		
		fsFilename = fsDirname + "/" + cbmFilename.ConvertThisToFilesystemFilename();
		//now make filename unique (if necessary) and open
		fsFilename = ::UniqueFileNameGenerator(fsFilename);
		out.open(fsFilename.c_str(), ios::out | ios::binary);
	}
	if (!out)
	{
		exc = CFException(CFException::FILE_OPEN_FAILED,
			__LINE__, __FILE__, fsFilename, "writing");
		throw exc;
	}

	(void)m_inStream.seekg((long)offset); //position to beginning
	if (!m_inStream)
	{
		exc = CFException(CFException::FILE_OPERATION_FAILED,
			__LINE__, __FILE__, m_filename, "positioning");
		throw exc;
	}

	//this format sucks big time!
	//the start address of the program is not stored in data
	//get it from the header instead and write it in front
	(void)out.put((char)m_fileRecords[nRecIdx].startAddressLow);
	(void)out.put((char)m_fileRecords[nRecIdx].startAddressHigh);
	char ch;
	unsigned int fileSize = 2; //consider the start address
	//now copy as much characters as needed
	if (length > 0)
	{
		//we need to copy a particular amount of characters
		while ((length-- > 0) && m_inStream.get(ch))
		{
			(void)out.put(ch);
			fileSize++;
		}
		if (m_inStream.bad())
		{
			exc = CFException(CFException::FILE_OPERATION_FAILED,
				__LINE__, __FILE__, m_filename, "reading");
			throw exc;
		}
		if (m_inStream.eof() || (length >= 0))
		{
			exc = CFException(CFException::TAPE_FILE_CORRUPTED,
				__LINE__, __FILE__, m_filename, cbmFilename);
			throw exc;
		}
	}
	else
	{
		//we have to read until end of stream
		while (m_inStream.get(ch))
		{
			(void)out.put(ch);
			fileSize++;
		}
		if (!m_inStream.eof())
		{
			exc = CFException(CFException::TAPE_FILE_CORRUPTED,
				__LINE__, __FILE__, m_filename, cbmFilename);
			throw exc;
		}
	}
	if (!out)
	{
		exc = CFException(CFException::FILE_OPERATION_FAILED,
			__LINE__, __FILE__, fsFilename, "reading");
		throw exc;
	}
	//everything worked well
	//set the info to put the extracted file into a disk image
	extractInfo.SetFileType(m_fileRecords[nRecIdx].c64DiskFileType);
	extractInfo.SetCbmFilename(cbmFilename);
	extractInfo.SetFilename(fsFilename);
	extractInfo.SetFileSize(fileSize);
}


void CFTapeImage::ExtractAllFiles(const string& dirname) throw (CFException)
{
	//iterate over all files within directory
	for (vector<tFileRecord>::size_type i = 0; i < m_fileRecords.size(); i++)
	{
		//extract suitable files
		if (m_fileRecords[i].entryType == NORMAL)
		{
			try {
			CFExtractedFileInfo fileInfo;
			ExtractFile(m_fileRecords[i], dirname, fileInfo);
			//if extraction worked out fine remeber all the info about
			//the extracted file to put it into a disk image later
			m_extractedFiles.push_back(fileInfo);
			}
			catch (CFException& actExc)
			{
				//handle missing directory not as warning
				if ((actExc.GetExceptionId() == CFException::UNEXPECTED_FILE_TYPE)
					|| (actExc.GetExceptionId() == CFException::FILE_STAT_FAILED))
					throw;
				actExc.WriteOutExceptionWithWarningHeader(cerr);
			}
		}
	}
}
