//////////////////////////////////////////////////////////////////////
//TargetD64 - C64 archive related conversion tool and emulator frontend
//////////////////////////////////////////////////////////////////////
//Copyright (C) 1998, 1999  Karlheinz Langguth klangguth@netscape.net
//
//This program is free software; you can redistribute it and/or
//modify it under the terms of the GNU General Public License
//as published by the Free Software Foundation; either version 2
//of the License, or (at your option) any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program; if not, write to the Free Software
//Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//////////////////////////////////////////////////////////////////////

//this file contains resources that are used from specialized images

#ifndef _IMAGE_HEADER
#define _IMAGE_HEADER

//concept of a CBM filename (flat, size limited, PETSCII, $a0 terminated)
class CFCbmFilename : public string
{
public:
	//fill whole string with $a0
	CFCbmFilename(void);
	//appends $a0 until maximum size reached
	//P1 I: PETSCII representation of filename (with trailing $a0)
	explicit CFCbmFilename(const string& cbmFilename);

public:
	//convert from host filesystem pathname
	//drop dirname, keep extension if possible, resize to maximum size
	//P1 I: host filesystem filename
	void ConvertThisFromFilesystemFilename(const string& fsFilename)
		 throw (CFException);
	//convert to host filesystem filename (no dirname)
	//R: converted filename
	const string& ConvertThisToFilesystemFilename(void) const;

public:
	static const unsigned int m_nMaxSize; //maximum size of cbm filename
	static const char m_termChar; //termination character, typical $a0
private:
	static char mapCbmToFilesys[256]; //character mapping table CBM->Host
	static char mapFilesysToCbm[256]; //character mapping Host->CBM
};


//concept of information on a CBM file
//some archives (e.g. lynx) contain this information and it has to be
//used to maintain the original CBM filename.
class CFFileInfoCBM
{
public:
	//these are the filetypes for directory entries with the real values
	typedef enum { DEL = 0, SEQ = 1, PRG = 2
		, USR = 3, REL = 4, UNKNOWN } tFiletype;

public:
	inline CFFileInfoCBM(void)
		: m_fileTypeForDisk(0xff) {}
public:
	//P1 I: cbm filetype (e.g. 82 for closed PRG)
	inline void SetFileType(const unsigned char fileTypeForDisk)
	{ m_fileTypeForDisk = fileTypeForDisk; }
	//P1 I: cbm filename
	inline void SetCbmFilename(const CFCbmFilename& filename)
	{ m_cbmFilename = filename; }
	//R: cbm filetype (e.g. 82 for closed PRG)
	inline unsigned char GetFileType(void) const
	{ return m_fileTypeForDisk; }
	//R: cbm filename
	inline const CFCbmFilename& GetCbmFilename(void) const
	{ return m_cbmFilename; }
	//R: true if members are still remain initial values
	inline bool TestInitialValues(void) const
	{ return (m_fileTypeForDisk == 0xff) && (m_cbmFilename == CFCbmFilename()); }

private:
	unsigned char m_fileTypeForDisk; //cbm filetype (e.g. 82 for closed PRG)
	CFCbmFilename m_cbmFilename; //cbm filename
};


//concept of information concerning an extracted file
//used to keep a 1:1 relation between host file and CFFileInfoCBM
class CFExtractedFileInfo : public CFFileInfoCBM
{
public:
	inline CFExtractedFileInfo(void)
		: m_fileSize(0) {}
public:
	//P1 I: host filename (complete pathname!)
	inline void SetFilename(const string& filename)
	{ m_filename = filename; }
	//P1 I: size of file in bytes
	inline void SetFileSize(const unsigned int size)
	{ m_fileSize = size; }
	//R: host filename (complete pathname!)
	inline const string& GetFilename(void) const
	{ return m_filename; }
	//R: size of file in bytes
	inline const unsigned int GetFileSize(void) const
	{ return m_fileSize; }

private:
	string m_filename; //host filename (complete pathname!)
	unsigned int m_fileSize; //size of file in bytes
};


//split pathname P1 into parts
//P1 I: pathname to split
//P2 O: dirname of P1 with trailing /
//P3 O: filename (without dirname and extension)
//P4 O: extension (with ., eg .lnx)
void SplitPathname(const string& pathName
	, string& dirname, string& filename, string& extension) throw (CFException);

//provide a unique pathname
//if already there, paste a unique infix between filename and extension
//until unique or maximum tries reached.
//If P2 is not NULL then the filename must ALSO be unique
//related to the filenames given in P2. The vector is
//shortened by the first element if this is found in the
//filesystem (it is not necessary to have a doubled storage)
//P2 is necessary for example when a D64 image is requested
//but NOT yet flushed into the filesystem. Still it is NOT
//allowed to have a second D64 image with the same name.
//P1 I: complete pathname of file to be made unique
//P2 IO: pointer to vector of already used filenames. Mind
//       that P2 is ALSO output param (see above).
//R: unique filename generated
string UniqueFileNameGenerator(const string& filename
	, vector<string> *usedFilenames = NULL) throw (CFException);

#ifdef _MSC_VER //just a little POSIX conformance for M$
#define S_ISDIR(a) ((a) & _S_IFDIR)
#define S_ISREG(a) ((a) & _S_IFREG)
#endif

#endif

