//////////////////////////////////////////////////////////////////////
//TargetD64 - C64 archive related conversion tool and emulator frontend
//////////////////////////////////////////////////////////////////////
//Copyright (C) 1998, 1999  Karlheinz Langguth klangguth@netscape.net
//
//This program is free software; you can redistribute it and/or
//modify it under the terms of the GNU General Public License
//as published by the Free Software Foundation; either version 2
//of the License, or (at your option) any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program; if not, write to the Free Software
//Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////
//COMPILE SWITCHES
// _MSC_VER
// indicates MS compiler
// _DEBUG
// for debug version which activates ASSERT and TRACE
// TD64_MODIFIED
// marks changes in foreign sources to fit into TargetD64
// must be set everywhere because also used for header files

#ifdef _MSC_VER
#pragma warning(disable:4786) //identifier truncation warning
#endif

#include <stdio.h>
#include <errno.h>
#ifdef _MSC_VER
#include <io.h>
#endif
#include <sys/stat.h>

#include <string>
#include <vector>
#include <iostream>
#include <strstream>
#include <fstream>
#ifdef _MSC_VER
#include <typeinfo.h>
#else
#include <typeinfo>
#endif

using namespace std;

#include "Exception.h"
#include "Image.h"
#include "DiskImage.h"
#include "TapeImage.h"
#include "LynxImage.h"
#include "Archive.h"
#include "MultiPart.h"
#include "C64Archive.h"
#include "Tracing.h"

#include "vdrive.h" //VICE interface


unsigned int CFD64Archive::BLOCKCAPACITY = 664;


//-------------------------------------------------------------------
//CFD64Archive
//-------------------------------------------------------------------


bool CFD64Archive::CheckFileIsOfClass(const string& filename) throw (CFException)
{
	string extension, invalid;

	::SplitPathname(filename, invalid, invalid, extension);
	if ((extension == ".d64") || (extension == ".D64"))
	{
		return true;	
	}
	return false;
}


void CFD64Archive::Extract(void) throw (CFException)
{
	TRBEGIN("CFD64Archive::Extract");

	const unsigned int D64_SIZE = 174848; //35 track D64
	const unsigned int D64_ERR_SIZE = 175531; //35 track D64 + error info
	const unsigned int D64EX_SIZE = 196608; //40 track D64
	const unsigned int D64EX_ERR_SIZE = 197376; //40 track D64 + error info

	//if already extracted or mother of all archives do no extraction
	if ((m_state == EXTRACTED) || (m_state == ROOT))//already done
		TRreturn;

	struct stat buf;

	if (stat(m_pathname.c_str(), &buf) != 0)
	{
		exc = CFException(CFException::FILE_STAT_FAILED,
			__LINE__, __FILE__, m_pathname, "", errno);
		throw exc;
	}
			
	//extract this archive into a new directory to build hirarchy
	m_workdir = NameAndCreateTmpDir();
	
	//extract by builtin diskimage handler
	switch (buf.st_size)
	{
	case D64_SIZE:
	case D64_ERR_SIZE:
		{
		CFDiskImageD64 extractDisk;
		extractDisk.Attach(m_pathname);
		m_bMoreFilesThanExtractedOnDisk = extractDisk.ExtractAllFiles(m_workdir);
		m_extractedFiles = extractDisk.GetVectorOfExtractedFiles();
		}	
		break;
	case D64EX_SIZE:
	case D64EX_ERR_SIZE:
		{
		CFDiskImageD64Ex extractDisk;
		extractDisk.Attach(m_pathname);
		m_bMoreFilesThanExtractedOnDisk = extractDisk.ExtractAllFiles(m_workdir);
		m_extractedFiles = extractDisk.GetVectorOfExtractedFiles();
		}
		break;
	default:
		exc = CFException(CFException::DISK_IMAGE_SIZE_INCORRECT,
			__LINE__, __FILE__, m_pathname, "size unexpected - not in (174848, 175531, 196608, 197376) bytes");
		throw exc;
	}

	TransitionTo(EXTRACTED);
	TRreturn;
}


void CFD64Archive::ProcessFilesInArchive(void) throw (CFException)
{
	TRBEGIN("CFD64Archive::ProcessFilesInArchive");
	//all processing done here - CFArchive not involved.

	string d64Name; //if the D64 is passed to emu this string contains name

	if (m_state != EXTRACTED)
		Extract(); //to process files we need extracted archive

	if (m_bMoreFilesThanExtractedOnDisk)
	{
		//it is indicated that we have archives in the D64 file that
		//could not be extracted, e.g. REL files => pass this D64 image to emu
		d64Name = BuildUniqueEmuImageName();
	}

	bool bAtLeastOneResolvableArchie = false;

	//extraction did store all extracted files - iterate over vector
	for (vector<CFExtractedFileInfo>::size_type i = 0; i < m_extractedFiles.size(); i++)
	{
		TRACE(<< "Processing file " << m_extractedFiles[i].GetFilename());
		unsigned int actualBlockSize = (m_extractedFiles[i].GetFileSize() + BLOCKSIZE_NETTO - 1)
			/ BLOCKSIZE_NETTO;
		//create the right object and queue it
		CFArchive *newEntry = CreateSubClassObject(m_extractedFiles[i].GetFilename()
			, actualBlockSize);
		m_files.push_back(newEntry);
		
		if (newEntry->Finalized())
		{
			//do NOT put final archive into d64 as they are from THIS d64
			if (d64Name.empty()) //check if already D64 emu image indicated
			{
				//we have a final archive => pass this D64 to emulator
				d64Name = BuildUniqueEmuImageName();
			}
		}
		else
		{
			//descent into archive
			bAtLeastOneResolvableArchie = true;
			newEntry->ProcessFilesInArchive();
		}
	}

	//if we had no resolvable archive on disk and the disk appears to be empty
	//still pass it to the emulator:
	//there are lot of games that do direct block access on disk without an
	//1541 compatible filesystem.
	if (d64Name.empty() && !bAtLeastOneResolvableArchie)
		d64Name = BuildUniqueEmuImageName();

	//rename the D64 image for the emulator use if indicated
	if (!d64Name.empty())
	{
		if (d64Name != m_pathname) //if file is named right do not rename
		{
#ifdef NOCOPY
			//copy source archive to temporary archive
			fstream in(m_pathname.c_str(), ios::in | ios::binary);
			if (!in)
			{
				exc = CFException(CFException::FILE_OPEN_FAILED,
					__LINE__, __FILE__, m_pathname, "reading");
				throw exc;
			}
	
			fstream out(d64Name.c_str(), ios::out | ios::binary);
			if (!out)
			{
				exc = CFException(CFException::FILE_OPEN_FAILED,
					__LINE__, __FILE__, d64Name, "write from scratch");
				throw exc;
			}	
			char ch;

			while (in.get(ch))
			{
				(void)out.put(ch);
			}

			if (in.bad())
			{
				exc = CFException(CFException::FILE_OPERATION_FAILED,
					__LINE__, __FILE__, m_pathname, "reading");
				throw exc;
			}
			if (!out)
			{
				exc = CFException(CFException::FILE_OPERATION_FAILED,
					__LINE__, __FILE__, d64Name, "writing");
				throw exc;
			}
#else

#ifdef _MSC_VER
			//only writable files can be removed for M$
			(void)::chmod(d64Name.c_str(), _S_IWRITE);
			//rename does not overwrite already existing files for M$
			//this may happen for non empty dir specified by -c
			(void)::remove(d64Name.c_str());
#endif
			if (::rename(m_pathname.c_str(), d64Name.c_str()))
			{
				exc = CFException(CFException::FILE_RENAME_FAILED,
					__LINE__, __FILE__, m_pathname, d64Name, errno);
				throw exc;
			}
#endif
		}
		//we have a new D64 for the emulator
		CFArchive *tmpArchive = CreateSubClassObject(d64Name);
		ms_emuArchives.push_back(tmpArchive);
	}
	else
	{
		if (m_files.size() == 0)
		{
			//we have not recognized a single file in archive
			//this will lead to a warning
			exc = CFException(CFException::ARCHIVE_EMPTY,
				__LINE__, __FILE__, GetPathname(), "");
			exc.WriteOutExceptionWithWarningHeader(cerr);
		}
	}
	TransitionTo(PROCESSED);
	TRreturn;
}


//-------------------------------------------------------------------
//CFX64Archive
//-------------------------------------------------------------------


bool CFX64Archive::CheckFileIsOfClass(const string& filename) throw (CFException)
{
	string extension, invalid;

	::SplitPathname(filename, invalid, invalid, extension);
	if ((extension == ".x64") || (extension == ".X64"))
	{
		return true;	
	}
	return false;
}


//Extract the X64 into D64
void CFX64Archive::Extract(void) throw (CFException)
{
	TRBEGIN("CFX64Archive::Extract");
	TRACE(<< "x64Image=" << m_pathname);
	
	//if already extracted or mother of all archives do no extraction
	if ((m_state == EXTRACTED) || (m_state == ROOT))//already done
		TRreturn;
			
	//extract this archive into a new directory to avoid double processing
	//(found by directory scan)
	m_workdir = NameAndCreateTmpDir();

	string d64Name = m_workdir + "/" + m_filename + ".d64";
	TRACE(<< "Extract into d64Image=" << d64Name);

	fstream in(m_pathname.c_str(), ios::in | ios::binary);
	if (!in)
	{
		exc = CFException(CFException::FILE_OPEN_FAILED,
			__LINE__, __FILE__, m_pathname, "reading");
		throw exc;
	}
	fstream out(d64Name.c_str(), ios::out| ios::binary);
	if (!out)
	{
		exc = CFException(CFException::FILE_OPEN_FAILED,
			__LINE__, __FILE__, d64Name, "write from scratch");
		throw exc;
	}	

	char ch;
	//Skip 64 bytes of a x64 archive
	for (int i = 0; i < 64; i++)
	{
		(void)in.get(ch);
	}
	while (in.get(ch))
	{
		(void)out.put(ch);
	}
	if (in.bad())
	{
		exc = CFException(CFException::FILE_OPERATION_FAILED,
			__LINE__, __FILE__, m_pathname, "reading");
		throw exc;
	}
	if (!out)
	{
		exc = CFException(CFException::FILE_OPERATION_FAILED,
			__LINE__, __FILE__, d64Name, "writing");
		throw exc;
	}

	TRreturn;
}


void CFX64Archive::ProcessFilesInArchive(void) throw (CFException)
{
	TRBEGIN("CFX64Archive::ProcessFilesInArchive");

	if (m_state != EXTRACTED)
		Extract(); //to process files we need extracted archive

	CFArchive *newEntry = 
		CreateSubClassObject(m_workdir + "/" + m_filename + ".d64");
	m_files.push_back(newEntry);
								
	//now process the contained d64 file
	newEntry->ProcessFilesInArchive();

	TransitionTo(PROCESSED);
	TRreturn;
}


//-------------------------------------------------------------------
//CFT64Archive
//-------------------------------------------------------------------


void CFT64Archive::Extract(void) throw (CFException)
{
	TRBEGIN("CFT64Archive::Extract");
	//if already extracted or mother of all archives do no extraction
	if ((m_state == EXTRACTED) || (m_state == ROOT))//already done
		TRreturn;
			
	//extract this archive into a new directory to build hirarchy
	m_workdir = NameAndCreateTmpDir();
	
	//extract by builtin tapeimage handler
	CFTapeImage extractTape;
	extractTape.Attach(m_pathname);
	extractTape.ExtractAllFiles(m_workdir);

	m_extractedFiles = extractTape.GetVectorOfExtractedFiles();

	TransitionTo(EXTRACTED);
	TRreturn;
}


void CFT64Archive::ProcessFilesInArchive(void) throw (CFException)
{
	TRBEGIN("CFT64Archive::ProcessFilesInArchive");
	//all processing done here - CFArchive not involved.

	if (m_state != EXTRACTED)
		Extract(); //to process files we need extracted archive

	try {
	//extraction did store all extracted files - iterate over vector
	for (vector<CFExtractedFileInfo>::size_type i = 0; i < m_extractedFiles.size(); i++)
	{
		TRACE(<< "Processing file " << m_extractedFiles[i].GetFilename());
		//calculate the blocksize of the file
		unsigned int actualBlockSize = (m_extractedFiles[i].GetFileSize() + BLOCKSIZE_NETTO - 1)
			/ BLOCKSIZE_NETTO;
		//create the right object and queue it
		CFArchive *newEntry = CreateSubClassObject(m_extractedFiles[i].GetFilename(), actualBlockSize);
		m_files.push_back(newEntry);
		
		if (newEntry->Finalized())
		{
			//store what you know about the CBM file which stands
			//behind the host file
			newEntry->SetFileInfoCBM(m_extractedFiles[i]);
		}
		//descent into archive
		newEntry->ProcessFilesInArchive();
	}
	} // END try
	catch (CFException&)
	{
		//flush potentially opened disk image with final archives of this archive
		CloseActualEmuImage();
		throw;
	}
	//flush potentially opened disk image with final archives of this archive
	CloseActualEmuImage();

	TransitionTo(PROCESSED);

	if (m_files.size() == 0)
	{
		//we have not recognized a single file in archive
		//this will lead to a warning
		exc = CFException(CFException::ARCHIVE_EMPTY,
			__LINE__, __FILE__, GetPathname(), "");
		exc.WriteOutExceptionWithWarningHeader(cerr);
	}

	TRreturn;
}


bool CFT64Archive::CheckFileIsOfClass(const string& filename) throw (CFException)
{
	string extension, invalid;

	::SplitPathname(filename, invalid, invalid, extension);
	if ((extension == ".t64") || (extension == ".T64"))
	{
		return true;	
	}
	return false;
}


//-------------------------------------------------------------------
//CFLnxArchive
//-------------------------------------------------------------------

static const string LNXPATTERN("USE LYNX TO DISSOLVE THIS FILE");

bool CFLnxArchive::CheckFileIsOfClass(const string& filename) throw (CFException)
{
	string extension, invalid;

	::SplitPathname(filename, invalid, invalid, extension);
	if ((extension == ".lnx") || (extension == ".LNX"))
	{
		return true;	
	}

	fstream in(filename.c_str(), ios::in | ios::binary);
	if (!in) //if we can not open the file no way to tell
	{
		return false;
	}
	
	char *puffer = new char[LNXPATTERN.size() + 1];
	ASSERT(puffer != NULL);
	memset(puffer, 0, LNXPATTERN.size() + 1);

	(void)in.seekg(0x38); //overread until pattern
	(void)in.read(puffer, (streamsize)LNXPATTERN.size());
	if (!in)
	{
		delete [] puffer;
		return false;
	}
	string pattern(puffer);
	
	delete [] puffer;
	
	if (pattern == LNXPATTERN)
	{
		return true;
	}
	//file is closed by destroying local variable in
	return false;
}

//TODO: There should be a common Superclass for the T64 and Lynx
//      at least ProcessFilesInArchive should be the quite the same

void CFLnxArchive::Extract(void) throw (CFException)
{
	TRBEGIN("CFLnxArchive::Extract");
	//if already extracted or mother of all archives do no extraction
	if ((m_state == EXTRACTED) || (m_state == ROOT))//already done
		TRreturn;

	//extract this archive into a new directory to build hirarchy
	m_workdir = NameAndCreateTmpDir();
	
	//extract by lynx image handler (which uses VICE)
	CFLynxImage extractLynx;
	extractLynx.Attach(m_pathname);
	extractLynx.ExtractAllFiles(m_workdir);

	m_extractedFiles = extractLynx.GetVectorOfExtractedFiles();

	TransitionTo(EXTRACTED);
	TRreturn;
}


void CFLnxArchive::ProcessFilesInArchive(void) throw (CFException)
{
	TRBEGIN("CFLnxArchive::ProcessFilesInArchive");

	if (m_state != EXTRACTED)
		Extract(); //to process files we need extracted archive

	try {
	//extraction did store all extracted files - iterate over vector
	for (vector<CFExtractedFileInfo>::size_type i = 0; i < m_extractedFiles.size(); i++)
	{
		TRACE(<< "Processing file " << m_extractedFiles[i].GetFilename());
		//calculate the blocksize of the file
		unsigned int actualBlockSize = (m_extractedFiles[i].GetFileSize() + BLOCKSIZE_NETTO - 1)
			/ BLOCKSIZE_NETTO;
		//create the right object and queue it
		CFArchive *newEntry = CreateSubClassObject(m_extractedFiles[i].GetFilename(), actualBlockSize);
		m_files.push_back(newEntry);
		
		if (newEntry->Finalized())
		{
			//store what you know about the CBM file which stands
			//behind the host file
			newEntry->SetFileInfoCBM(m_extractedFiles[i]);
		}
		//descent into archive
		newEntry->ProcessFilesInArchive();
	}
	} // END try
	catch (CFException&)
	{
		//flush potentially opened disk image with final archives of this archive
		CloseActualEmuImage();
		throw;
	}
	//flush potentially opened disk image with final archives of this archive
	CloseActualEmuImage();

	TransitionTo(PROCESSED);

	if (m_files.size() == 0)
	{
		//we have not recognized a single file in archive
		//this will lead to a warning
		exc = CFException(CFException::ARCHIVE_EMPTY,
			__LINE__, __FILE__, GetPathname(), "");
		exc.WriteOutExceptionWithWarningHeader(cerr);
	}
	TRreturn;
}


//-------------------------------------------------------------------
//CFZipC64Archive
//-------------------------------------------------------------------


bool CFZipC64Archive::CheckFileIsOfClass(const string& filename)
	throw (CFException)
{
	string zipCode, invalid;

	::SplitPathname(filename, invalid, zipCode, invalid);
	//get the first two characters of the filename only
	if (zipCode.size() > 2)
		zipCode.resize(2);

	if ((zipCode == "1!") || (zipCode == "2!")
		|| (zipCode == "3!")|| (zipCode == "4!"))
	{
		return true;
	}
	return false;
}


CFZipC64Archive::CFZipC64Archive(const string& path, const unsigned int blockSize)
	throw (CFException)
: CFC64Archive(path, blockSize)
, CFMultiPart(4)
{
	TRBEGIN("CFZipC64Archive::CFZipC64Archive");

	//set the part used for construction to "available"
	string prefix = m_filename.substr(0, 1);
	ASSERT((prefix > "0") && (prefix < "5"));
	unsigned int nIndex = (unsigned char)(prefix[0] - '1');
	(void)SetPartAvailable(nIndex);
	//set the pathname of the part used for construction
	SetPartPathname(nIndex, m_pathname);
	TRreturn;
};


void CFZipC64Archive::Extract(void) throw (CFException)
{
	TRBEGIN("CFZipC64Archive::Extract");

	//if already extracted or mother of all archives do no extraction
	if ((m_state == EXTRACTED) || (m_state == ROOT))//already done
		TRreturn;
			
	//extract this archive into a new directory
	m_workdir = NameAndCreateTmpDir();
	
	string filename = GetFilename();
	//strip away the [1-4]!
	filename = filename.substr(2, filename.length() - 2);
	string outNameD64 = m_workdir + "/" + filename + ".d64";

	ASSERT(GetPartNum() == 4);
	const char *partNames[4];
	for (unsigned int i = 0; i < GetPartNum(); i++)
	{
		partNames[i] = GetPartPathname(i).c_str();
	}
	FILE *outFileDescr;
	//open the output D64 image in C style because access
	//to foreign sources
	if ((outFileDescr = fopen(outNameD64.c_str(), "wb")) == NULL)
	{
		exc = CFException(CFException::FILE_OPEN_FAILED,
			__LINE__, __FILE__, outNameD64, "writing", errno);
		throw exc;
	}
	if (zcreate_cmd(outFileDescr, partNames) != FD_OK)
	{
		(void)::fclose(outFileDescr);
		(void)::remove(outNameD64.c_str());
		exc = CFException(CFException::C64ZIP_EXTRACT_FAILED,
			__LINE__, __FILE__, GetPathname(), "");
		throw exc;
	}
	::fclose(outFileDescr);

	TransitionTo(EXTRACTED);	
	TRreturn;
}


void CFZipC64Archive::ProcessFilesInArchive(void) throw (CFException)
{
	TRBEGIN("CFZipC64Archive::ProcessFilesInArchive");

	bool complete =	TestCompleteness();
	if (!complete)
	{
		//encountered an incomplete multipart - search for missing parts

		//now we have to cope with archives that exist of multiparts
		//like the C64Zip 1!,2!,3!,4!
		//they may be spread all over one layer of the archive tree
		//
		//check all archives on a layer for belonging multipart
		//this means e.g
		//                    1
		//              11   12   13
		//            a b c  d   e f g   <---- check all on this layer

		TRACE(<< "ZipC64 not yet complete - search for missing parts");
		//start search from the parent archive - this archive has no children yet
		unsigned int level = 0;
		CFArchive* pTopArchive = const_cast<CFArchive *>(GetParentArchivePointer());
		//we search from the closest neighbors to the farest
		//but only for nodes on THE SAME level
		while ((pTopArchive != NULL) && !complete)
		{
			//search for counterparts of this archive
			complete =
			pTopArchive->TraverseAllArchivesOfLevelSearchingMultipart(*this
				, level);
			//climb up to reach more archives on this layer
			pTopArchive =
				const_cast<CFArchive *>(pTopArchive->GetParentArchivePointer());
			level++;
		}
	}
	if (!complete)
	{
		TRACE(<< "ZipC64 could not be completed");
		TRreturn;
	}

	Extract(); //we need this archive to be extracted to produce d64

	//mind that Extract() may produce any file supported by TD64
	//c1541 for example will produce X64, which is no problem
	//it is better to produce any C64 image than C64 raw files
	//then the filenames are kept because there are not converted to the host filesystem
	CFArchive::ProcessFilesInArchive();

	TRreturn;
}

    
bool CFZipC64Archive::CheckBelongingAndMergeIfBelongs(const CFArchive& archive)
	throw (CFException)
{
	TRBEGIN("CFZipC64Archive::CheckBelongingAndMergeIfBelongs");
	TRACE(<< "Candidate " << archive.GetFilename());
	TRACE(<< "This archive " << m_filename);
	
	//May only belong here if it is a C64Zip Archive
	if (typeid(archive) != typeid(CFZipC64Archive))
	{
		TRACE(<< "Does not belong here because no C64Zip");
		TRreturn false;
	}

	CFMultiPart const *pMultiPart = dynamic_cast<CFMultiPart const *>(&archive);
	ASSERT(pMultiPart);

	string pathname = archive.GetPathname();	
	ASSERT((int)string::npos == -1) //assure that next line works even if there is no /
	unsigned int pos = pathname.find_last_of("/");
	string file = pathname.substr(pos + 1, pathname.length() - pos - 1);
	
	ASSERT(file.length() > 1)
	string postfix = file.substr(2, file.length() - 2);

	if (postfix == m_filename.substr(2, file.length() - 2) + m_extension)
	{
		//check for those parts that are in the other archive
		for (unsigned int i = 0; i < 4; i++)
		{
			if (pMultiPart->TestPartAvailable(i))
			{
				TRACE(<< "Merging part with index: " << i);
				if (SetPartAvailable(i))
				{
					exc = CFException(CFException::MULTIPLE_PART_OF_MULTIPART_ARCHIVE,
						__LINE__, __FILE__, m_pathname, "");
					throw exc; //has been already there
				}
				else
				{
					//set the part successfully avail - now remember pathname
					SetPartPathname(i, pMultiPart->GetPartPathname(i));
				}
			}
		}
		TRACE(<< "Belongs here");
		TRreturn true;
	}
	TRACE(<< "Does not belong here");
	TRreturn false;
}


void CFZipC64Archive::CopyThisArchiveToDirectory(const string& directory)
throw (CFException)
{
	TRBEGIN("CFZipC64Archive::CopyThisArchiveToDirectory");
	TRACE(<< "directory " << directory);

	//strip away directory of archive
	ASSERT((m_filename.length() > 2) && (m_filename[1] == '!'))

	//try to copy all four parts
	string storeArchive = CFArchive::m_filename;
	for (int i = 1; i < 5; i++)
	{
		m_filename[0] = (char)('0' + i);
		//attention: we need all four parts, else there is an exception
		CFArchive::CopyThisArchiveToDirectory(directory);
	}
	m_filename = storeArchive;
	TRreturn;
}


//-------------------------------------------------------------------
//CFPc64Archive
//-------------------------------------------------------------------


bool CFPc64Archive::CheckFileIsOfClass(const string& filename)
{
	fstream in(filename.c_str(), ios::in | ios::binary);
	if (!in) //if we can not open the file no way to tell
	{
		return false;
	}
	
	char puffer[8];
	(void)in.read(puffer, sizeof(puffer));
	//archive is solely recognized by string pattern
	if (!strcmp(puffer, "C64File"))
	{
		return true;
	}
	return false;
}


CFPc64Archive::CFPc64Archive(const string& path, const unsigned int blockSize /*= 0*/)
	throw (CFException)
	: CFC64Archive(path, blockSize)
	, m_pInStream(NULL)
	, m_contentFileByteSize(0)
{
	//bit within dir entry set when file is closed
	const unsigned char closed = 0x80;
	//bit within dir entry set when file is locked
	const unsigned char locked = 0x40;
	//this is the filetype recognized from extension to be put into dir entry
	unsigned char nFileType = CFFileInfoCBM::PRG | closed;

	string extension = m_extension;
	//make chars lower case for simple comparison
	for (string::size_type i = 0; i < extension.size(); i++)
		extension[i] = (char)tolower(extension[i]);

	if (extension == ".p00")
	{
		nFileType = CFFileInfoCBM::PRG | closed;
	}
	else
	{
		if (extension == ".s00")
		{
			nFileType = CFFileInfoCBM::SEQ | closed;
		}
		else
		{
			if (extension == ".u00")
			{
				nFileType =  CFFileInfoCBM::USR | closed;
			}
			else
			{
				if (extension == ".d00")
				{
					//DEL files must be locked to appear
					nFileType = CFFileInfoCBM::DEL | closed | locked;
				}
				else
				{
					if (extension == ".r00")
					{
						nFileType = CFFileInfoCBM::REL | closed;
					}
				}
			}
		}
	}
	//now get the C64filename from the file header and leave file open
	//for processing
	m_pInStream = new ifstream(m_pathname.c_str(), ios::in | ios::binary);
	ASSERT(m_pInStream != NULL);
	if (!(*m_pInStream))
	{
		exc = CFException(CFException::FILE_OPEN_FAILED,
			__LINE__, __FILE__, m_pathname, "reading");
		throw exc;
	}
	//skip 8 bytes of file type pattern
	(void)m_pInStream->seekg(0x8);
	char puffer[17];
	//read 17 bytes c64 filename padded with '\0'
	(void)m_pInStream->read(puffer, sizeof(puffer));
	if (!(*m_pInStream))
	{
		m_pInStream->close();
		exc = CFException(CFException::FILE_OPERATION_FAILED,
			__LINE__, __FILE__, m_pathname, "reading");
		throw exc;
	}
	//now set the CBM fileinfo for this archive
	m_fileInfoCBM.SetFileType(nFileType);
	m_fileInfoCBM.SetCbmFilename(CFCbmFilename(string(puffer)));
}


void CFPc64Archive::Extract(void) throw (CFException)
{
	TRBEGIN("CFPc64Archive::Extract");
	//if already extracted or mother of all archives do no extraction
	if ((m_state == EXTRACTED) || (m_state == ROOT))//already done
		TRreturn;
			
	//extract this archive into a new directory to avoid double processing
	//(found by directory scan)
	m_workdir = NameAndCreateTmpDir();

	ASSERT(m_pInStream != NULL);
	ASSERT(m_pInStream->rdbuf()->is_open()); //assure that the stream is opened
	(void)m_pInStream->seekg(26); //position to start of raw data
	//open the output file - filename due to CBM filename (maybe we
	//can keep extensions this way like .lnx)
	m_contentFilename = m_workdir + "/"
		+ GetFileInfoCBM().GetCbmFilename().ConvertThisToFilesystemFilename();
	ofstream out(m_contentFilename.c_str(), ios::out | ios::binary);
	if (!out)
	{
		m_pInStream->close();
		exc = CFException(CFException::FILE_OPEN_FAILED,
			__LINE__, __FILE__, m_contentFilename, "writing");
		throw exc;
	}

	m_contentFileByteSize = 0;
	char ch;
	while (m_pInStream->get(ch))
	{
		m_contentFileByteSize++; //one more byte written to output file
		(void)out.put(ch);
	}

	if (m_pInStream->bad())
	{
		m_pInStream->close();
		exc = CFException(CFException::FILE_OPERATION_FAILED,
			__LINE__, __FILE__, m_pathname, "reading");
		throw exc;
	}
	//close the file otherwise M$ can not remove this file while cleaning up!
	m_pInStream->close();

	if (!out)
	{
		exc = CFException(CFException::FILE_OPERATION_FAILED,
			__LINE__, __FILE__, m_contentFilename, "writing");
		throw exc;
	}
	
	TransitionTo(EXTRACTED);
	TRreturn;
}


void CFPc64Archive::ProcessFilesInArchive(void) throw (CFException)
{
	TRBEGIN("CFPc64Archive::ProcessFilesInArchive");

	if (m_state != EXTRACTED)
		Extract(); //to process files we need extracted archive

	ASSERT(!m_contentFilename.empty());
	//now create the real c64 file contained as a BROTHER of this archive
	//this is good for putting those files into the same D64 image.
	//Otherwise each Pc64 file would lead to a single D64 image which is
	//nonsense.
	CFArchive *newEntry = GetParentArchivePointer()->
		CreateSubClassObject(m_contentFilename
		, (m_contentFileByteSize + BLOCKSIZE_NETTO - 1) / BLOCKSIZE_NETTO);
	//propagate the CBM fileinfo to brother
	newEntry->SetFileInfoCBM(GetFileInfoCBM());
	//add this newly created archive to the others
	const_cast<CFArchive *>(GetParentArchivePointer())->AddNewArchiveToChildren(newEntry);

	//now process the contained c64 file
	newEntry->ProcessFilesInArchive();

	TransitionTo(PROCESSED);
	TRreturn;
}


//-------------------------------------------------------------------
//CFFinalArchive
//-------------------------------------------------------------------


void CFFinalArchive::Extract(void) throw (CFException)
{
	TRBEGIN("CFFinalArchive::Extract");
	//nothing to do, because we have a final archive
	TransitionTo(EXTRACTED);
	TRreturn;
}


void CFFinalArchive::ProcessFilesInArchive(void) throw (CFException)
{
	TRBEGIN("CFFinalArchive::ProcessFilesInArchive");

	//true if a disk image has been created in this call
	bool bDiskImageFreshlyCreated = false;
	//this final archive is related to a D64 file located in its parent archive
	CFDiskImageD64*& pDiskImage = const_cast<CFArchive *>(GetParentArchivePointer())
		->GetReferenceToD64EmuImagePointer();
	//if new diskimage is indicated create a new diskimage
	if (!pDiskImage)
	{
		string d64Name = GetParentArchivePointer()->BuildUniqueEmuImageName();
		pDiskImage = new CFDiskImageD64(d64Name);
		ASSERT(pDiskImage != NULL);
		bDiskImageFreshlyCreated = true;
	}
	try {
	if (ReadBlockSize() > CFD64Archive::BLOCKCAPACITY)
	{
		exc = CFException(CFException::FILE_TOO_BIG_FOR_D64
			,__LINE__, __FILE__, GetPathname()
			, CFException::IntToString((int)ReadBlockSize()));
		throw exc;
	}		
	if (ReadBlockSize() > pDiskImage->GetFreeBlockCountDisk())
	{
		//if file won't fit create new emu image
		//will delete pDiskImage
		const_cast<CFArchive *>(GetParentArchivePointer())->CloseActualEmuImage();
		//create a new D64 file
		string d64Name = GetParentArchivePointer()->BuildUniqueEmuImageName();
		ASSERT(pDiskImage == NULL);
		pDiskImage = new CFDiskImageD64(d64Name);
		ASSERT(pDiskImage != NULL);
		bDiskImageFreshlyCreated = true;
	}
	//insert this final archive into diskimage (keep CBM fileinfo if given)
	pDiskImage->InsertFile(GetPathname(), GetFileInfoCBM());
	}
	//if insertion of first file went wrong delete the whole image
	catch (CFException& actExc)
	{
		if (bDiskImageFreshlyCreated)
		{
			//file not inserted -> rollback newly opened disk image
			delete pDiskImage;
			pDiskImage = NULL;
		}
		if (actExc.GetExceptionId() == CFException::FILE_TOO_BIG_FOR_D64)
		{
			//file too big will only produce warning - else ignored
			actExc.WriteOutExceptionWithWarningHeader(cerr);
		}
		else
		{
			//any other error is fatal
			throw;
		}
	}

	TransitionTo(PROCESSED);
	TRreturn;
}


