//////////////////////////////////////////////////////////////////////
//TargetD64 - C64 archive related conversion tool and emulator frontend
//////////////////////////////////////////////////////////////////////
//Copyright (C) 1998, 1999  Karlheinz Langguth klangguth@netscape.net
//
//This program is free software; you can redistribute it and/or
//modify it under the terms of the GNU General Public License
//as published by the Free Software Foundation; either version 2
//of the License, or (at your option) any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program; if not, write to the Free Software
//Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////
//COMPILE SWITCHES
// _MSC_VER
// indicates MS compiler
// _DEBUG
// for debug version which activates ASSERT and TRACE
// TD64_MODIFIED
// marks changes in foreign sources to fit into TargetD64
// must be set everywhere because also used for header files

#ifdef _MSC_VER
#pragma warning(disable:4786) //identifier truncation warning
#endif

#include <stdlib.h>
#include <string.h>
#include <errno.h>
#ifdef _MSC_VER
#include <windows.h>
#include <direct.h>
#include <io.h>
#else
#include <stdio.h>
#include <sys/types.h>
#include <fcntl.h>
#include <unistd.h>
#include <dirent.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>

#include <algorithm>
#include <map>
#include <vector>
#include <string>
#include <iomanip>
#include <iostream>
#include <fstream>
#include <strstream>
#ifdef _MSC_VER
#include <typeinfo.h>
#else
#include <typeinfo>
#endif

using namespace std;

#include "Options.h"
#include "Profile.h"
#include "TargetD64.h"
#include "Exception.h"
#include "Image.h"
#include "DiskImage.h"
#include "Archive.h"
#include "HostArchive.h"
#include "MultiPart.h"
#include "C64Archive.h"
#include "Tracing.h"


string CFArchive::ms_tmpdir;
string CFArchive::ms_emuArchiveBasename;
unsigned int CFArchive::ms_emuArchiveCount = 0;
string CFArchive::ms_emuImageDir;
string CFArchive::ms_miscDir;
bool CFArchive::ms_bEmuImageCleanup = true;
bool CFArchive::ms_bEmuImageNameDueOriginal = false;
bool CFArchive::ms_bCleanup = true;

vector<CFArchive *> CFArchive::ms_emuArchives;


//a little POSIX conformance for M$
#ifdef _MSC_VER

//provide UNIX scan directory functions
typedef HANDLE DIR;

struct dirent
{
	char d_name[MAX_PATH];
};

static DIR *opendir(const char *name);
static int closedir(DIR *dir);
static struct dirent *readdir(DIR *dir);

#endif


string& CFArchive::BuildHostConformFilename(const string& filename)
{
	static string retFilename;
	
	retFilename = filename;
	for (string::size_type i = 0; i < filename.size(); i++)
	{
		//replace non usable characters by _
		if ((retFilename[i] == '\'')
			|| (retFilename[i] == ' ')
			|| (retFilename[i] == '&')
			|| (retFilename[i] == ';'))
			retFilename[i] = '_';
	}
	return retFilename;
}


#ifdef _MSC_VER

//commands are executed in batchfiles for M$
//% characters have to be auoted by percent characters in batchfiles
static const string& QuotePercentCharacter(const string& in)
{
	static string retString;

	retString = in;

	//ATTENTION: % character in M$ batchfiles must be % quoted (%%)
	string::size_type pos = 0;
	while ((pos = retString.find("%", pos)) != string::npos)
	{
		//from the left to the percent (included)
		string leftEqualPercent = retString.substr(0, pos + 1);
		//from the percent (included) to the right
		string rightEqualPercent = retString.substr(pos
			, retString.size() - pos);
		//this is the percent doubling
		retString = leftEqualPercent + rightEqualPercent;
		//make sure that originally found percent and appended
		//percent are not found
		pos += 2;
		if (pos >= retString.size())
			pos = string::npos;
	}
	return retString;
}

#endif


//For Linux:
//replace %s<index> with replacement
//For Windows:
//replace %s<index> with "short filename" from UNIX filename converted replacement
//replace %l<index> with "long filename"  from UNIX filename converted replacement
//replace %d<index> with the drive of from UNIX filename converted replacement
//this stuff is confusing:
//a prefix of "basename " for P3 marks that basename of filename is wanted
//this is needed to get a short and long filename version for Win32
//otherwise we could have created the basename before calling

#ifdef _MSC_VER
//we have batchfile execution - quote % to %%
#define REPLACEMENT(a) (QuotePercentCharacter(a))
#else
#define REPLACEMENT(a) (a)
#endif

static void ReplaceHelperFunction(string& command, char index, const string& replacement)
	throw (CFException) //e.g. indirect via SplitPathname
{
	string::size_type pos = 0;
	string filenameToBuildBasenameFrom;

	//check if somebody has requested a filename's basename
	if (replacement.substr(0, 9) == "basename ")
	{
		filenameToBuildBasenameFrom = replacement.substr(9, replacement.size() - 9);
	}
#ifndef _MSC_VER

	while ((pos = command.find((string)"%s" + index, pos)) != string::npos)
	{
		string tmp;
		if (!filenameToBuildBasenameFrom.empty())
		{
			//basename of filename requested
			string directory, filename, extension;
			::SplitPathname(filenameToBuildBasenameFrom
				, directory, filename, extension);
			tmp = command.substr(0, pos) + filename + extension;
		}
		else
		{
			tmp = command.substr(0, pos) + replacement;
		}
		//add non substituted command and search on
		command = tmp + command.substr(pos + 3, command.size() - pos - 3);
		//do not consider substitution for replacement
		pos = ((tmp.size() == 0) ? 1 : tmp.size()) - 1;
	}

#else

	//scan for the substitution character
	while ((pos = command.find("%", pos)) != string::npos)
	{
		if (command.substr(pos + 1, 2) == (string)"l" + index)
		{
			string tmp;
			if (!filenameToBuildBasenameFrom.empty())
			{
				//basename of filename requested
				string directory, filename, extension;
				::SplitPathname(CFArchive::ConvertFilenameUnixToWin(filenameToBuildBasenameFrom)
					, directory, filename, extension);
				tmp = filename + extension;
			}
			else
			{
				tmp = CFArchive::ConvertFilenameUnixToWin(replacement);
			}
			//add non substituted command and search on
			command = command.substr(0, pos) + REPLACEMENT(tmp)
				+ command.substr(pos + 3, command.size() - pos - 3);
			//do not consider substitution for replacement
			pos += tmp.size();
			if (pos >= command.size())
				pos = string::npos;
		}
		else
		{
			if (command.substr(pos + 1, 2) == (string)"s" + index)
			{
				string tmp;
				if (!filenameToBuildBasenameFrom.empty())
				{
					//basename of filename requested
					string directory, filename, extension;
					::SplitPathname(CFArchive::ConvertFilenameUnixToWinShort(filenameToBuildBasenameFrom)
						, directory, filename, extension);
					tmp = filename + extension;
				}
				else
				{
					tmp = CFArchive::ConvertFilenameUnixToWinShort(replacement);
				}
				//add non substituted command and search on
				command = command.substr(0, pos) + REPLACEMENT(tmp)
					+ command.substr(pos + 3, command.size() - pos - 3);
				//do not consider substitution for replacement
				pos += tmp.size();
				if (pos >= command.size())
					pos = string::npos;
			}
			else
			{
				if (command.substr(pos + 1, 2) == (string)"d" + index)
				{
					string drive;
					if (!filenameToBuildBasenameFrom.empty())
					{
						drive = CFArchive::ConvertFilenameUnixToWin(filenameToBuildBasenameFrom);
					}
					else
					{
						drive = CFArchive::ConvertFilenameUnixToWin(replacement);
					}
					if ((drive.size() > 1) && (drive[1] == ':'))
					{
						drive.resize(2);
					}
					else
					{
						drive.resize(0);
					}
					string tmp = command.substr(0, pos) + drive;
					//add non substituted command and search on
					command = REPLACEMENT(tmp) + command.substr(pos + 3, command.size() - pos - 3);
					//do not consider substitution for replacement
					pos = ((tmp.size() == 0) ? 1 : tmp.size()) - 1;
				}
				else
				{
					//we have found % but there is nothing to be replaced
					pos++; //same % should not be found again
				}
			}
		}
	}

#endif
}


static void BuildAbsoluteDirectoryPath(string &dirPath) throw (CFException)
{
	TRBEGIN("BuildAbsoluteDirectoryPath");
	TRACE(<< "Dirpath: " << dirPath);
	char *tmpWd = getcwd(NULL, 0); //allocs memory!
	if (tmpWd == NULL)
	{
		exc = CFException(CFException::GET_CURRENT_DIRECTORY_FAILED,
			__LINE__, __FILE__, "", "", errno);
		throw exc;
	}
	string oldWd(tmpWd);
	free(tmpWd);
	
	if (::chdir(dirPath.c_str()))
	{
		exc = CFException(CFException::CHANGE_DIRECTORY_FAILED,
			__LINE__, __FILE__, dirPath, "", errno);
		throw exc;
	}

	tmpWd = getcwd(NULL, 0); //allocs memory!
	if (tmpWd == NULL)
	{
		exc = CFException(CFException::GET_CURRENT_DIRECTORY_FAILED,
			__LINE__, __FILE__, "", "", errno);
		throw exc;
	}
	dirPath = tmpWd;
	free(tmpWd);

	//restore old directory
	if (::chdir(oldWd.c_str()))
	{
		exc = CFException(CFException::CHANGE_DIRECTORY_FAILED,
			__LINE__, __FILE__, oldWd, "", errno);
		throw exc;
	}
	TRreturn;
}


void CFArchive::ReadSettings(void) throw (CFException)
{
	TRBEGIN("CFArchive::ReadSettings");

	string value;
	//-----------------------------------------------------------
	//Profile settings
	//-----------------------------------------------------------
	const CFProfile *pProfile = CFProfile::GetAppProfile();
	ASSERT(pProfile != NULL);
	bool flag;
	if (pProfile->ReadProfileValue(PRO_OPT_NOCLEANUP.first, PRO_OPT_NOCLEANUP.second, flag)
		&& flag)
	{
		ms_bCleanup = false;
	}
	if (pProfile->ReadProfileValue(PRO_OPT_KEEP.first, PRO_OPT_KEEP.second, flag)
		&& flag)
	{
		ms_bEmuImageNameDueOriginal = true;
	}
	string tmpdir;
	(void)pProfile->ReadProfileValue(PRO_ENV_TMPDIR.first, PRO_ENV_TMPDIR.second, tmpdir);

	//-----------------------------------------------------------
	//Command line options
	//-----------------------------------------------------------
	const CFOptions *pOpt = CFOptions::GetAppOptions();
	ASSERT(pOpt != NULL);
	ms_bCleanup &= !pOpt->QueryOption(OPT_NO_CLEANUP.first, value);	//no cleanup
	ms_bEmuImageNameDueOriginal |= pOpt->QueryOption(OPT_KEEP_FILENAME.first, value);

	//-----------------------------------------------------------
	//Environment variables
	//-----------------------------------------------------------
	//either use environment variable TMPDIR or /tmp
	static char* env = ::getenv("TMPDIR");
#ifdef _MSC_VER
	if (!env)
	{
		env = ::getenv("TMP");
		if (!env)
		{
			env = ::getenv("TEMP");
		}
	}
#endif
	if (env != NULL)
	{
		tmpdir = env;
	}
	else if (tmpdir.empty())
	{
#ifdef _MSC_VER
		tmpdir = "c:\\temp";
#else
		tmpdir = "/tmp";
#endif
	}

	//now eval the absolute path - ATTENTION: must be absolute
	::BuildAbsoluteDirectoryPath(tmpdir);

	//now that the temp dir is determined set it
	ms_tmpdir = tmpdir;

	//-----------------------------------------------------------
	//Generic archives from environment and profile
	//-----------------------------------------------------------

	//check for generic archive extraction commands
	//Search for pairs of
	//TD64_GENERIC_CMDx and TD64_GENERIC_EXTENSIONx
	//must be in numerical order 1..n - otherwise not found
	int i = 1;
	while (i > 0)
	{
		strstream ret;
		ret << i << '\0';
		string genericCommand = (string)"TD64_GENERIC_CMD" + ret.str();
		string genericExtension = (string)"TD64_GENERIC_EXT" + ret.str();
		char *strCmd;
		if ((strCmd = ::getenv(genericCommand.c_str())) != NULL)
		{
			TRACE(<< genericCommand << ": " << strCmd);
			char *strExt;
			if ((strExt = ::getenv(genericExtension.c_str())) != NULL)
			{
				TRACE(<< genericExtension << ": " << strExt);
				//user defined generic archive encountered - queue it
				CFGenericArchive userDefinedGenericArchiv(
					"", strCmd, strExt, 0);
				//mind that vector take a copy of the auto class
				//variable userDefinedGenericArchiv
				CFGenericArchive::ms_genericArchives
					.push_back(userDefinedGenericArchiv);
				i++;
			}
			else
			{
				exc = CFException(CFException::GENERIC_EXTENSION_MISSING_ERROR,
					__LINE__, __FILE__, genericCommand, genericExtension);
				exc.WriteOutExceptionWithWarningHeader(cerr);
				i = -1;
			}
		}
		else
			i = -1;
	}

	//now read generic archive definitions from profile

	vector<string> genericArchives;
	pProfile->ReadProfileValue(PRO_GENERIC_ARCHIVES.first, PRO_GENERIC_ARCHIVES.second, genericArchives);
	for (vector<string>::size_type j = 0; j < genericArchives.size(); j++)
	{
		string strExt, strCmd;
		string::size_type nSeparator = genericArchives[j].find(',');
		if (nSeparator != string::npos)
		{
			strExt = genericArchives[j].substr(0, nSeparator);
			strCmd = genericArchives[j].substr(nSeparator + 1, genericArchives[j].size() - nSeparator - 1);
			TRACE(<< "Generic archive extension from profile: " << ": " << strExt);
			TRACE(<< "Generic archive command from profile: " << ": " << strCmd);
			//user defined generic archive encountered - queue it
			CFGenericArchive userDefinedGenericArchiv(
				"", strCmd, strExt, 0);
			//mind that vector take a copy of the auto class
			//variable userDefinedGenericArchiv
			CFGenericArchive::ms_genericArchives
				.push_back(userDefinedGenericArchiv);
		}
		else
		{
			string param1 = "Syntax error in generic archive definition: "
				+ genericArchives[j];
			exc = CFException(CFException::PROFILE_ERROR,
				__LINE__, __FILE__, param1, "");
			exc.WriteOutExceptionWithWarningHeader(cerr);
		}
	}

	TRreturn;
}


//P2-P6 replacement values
//      ATTENTION: "basename " prefix will lead to basename
//this stuff is confusing:
//a prefix of "basename " for P2-P6 marks that basename of filename is wanted
//this is needed to get a short and long filename version for Win32
//otherwise we could have created the basename before calling
string& CFArchive::ReplacePlaceHoldersByStrings(
	const string& inString
	, const string& s1
	, const string& s2
	, const string& s3 /*= ""*/
	, const string& s4 /*= ""*/
	, const string& s5 /*= ""*/) throw (CFException)
{	
	static string retString;

	retString = inString;
	//replace placeholders by real pathes
	::ReplaceHelperFunction(retString, '1', s1);
	::ReplaceHelperFunction(retString, '2', s2);
	if (s3.empty())
		return retString;
	::ReplaceHelperFunction(retString, '3', s3);
	if (s4.empty())
		return retString;
	::ReplaceHelperFunction(retString, '4', s4);
	if (s5.empty())
		return retString;
	::ReplaceHelperFunction(retString, '5', s5);
	return retString;
}


CFArchive::CFArchive() : m_pParentArchive(NULL)
	, m_state(CREATED)
	, m_final(false)
	, m_blockSize(0)
	, m_pDiskImage(NULL)
	//-1 is greatest unsigned int
	, m_nRequestOrderIndex((unsigned int)-1)
{
	TRBEGIN("CFArchive::CFArchive");
	TRreturn;
}


//SetPathComponents may throw exception
CFArchive::CFArchive(const string& pathname, const unsigned int blockSize /*= 0*/)
	throw (CFException)
	: m_pParentArchive(NULL)
	, m_state(CREATED)
	, m_final(false)
	, m_workdir(ms_tmpdir)
	, m_blockSize(blockSize)
	, m_pDiskImage(NULL)
	//-1 is greatest unsigned int
	, m_nRequestOrderIndex((unsigned int)-1)
{
	TRBEGIN("CFArchive::CFArchive");
	TRACE(<< "pathname=" << pathname << " blockSize=" << blockSize);

	SetPathComponents(pathname);	

	TRreturn;
}


CFArchive::~CFArchive()
{
	//ATTENTION:
	//the following assumes ALL archives created by CreateSubClassObject() or
	//at least a new operator.
	for (vector<CFArchive *>::size_type i = 0; i < m_files.size(); i++)
	{
		delete m_files[i];
	}
	//for the ROOT archive cleanup the emulator archives
	if (m_state == ROOT)
	{
		for (vector<CFArchive *>::size_type j = 0; j < ms_emuArchives.size(); j++)
		{
			delete ms_emuArchives[j];
		}
	}
	ASSERT(m_pDiskImage == NULL); //should have been cleaned before
}


void CFArchive::InitRootArchiveAndEnvironment(const string& pathname, const string& emuImageDir)
	throw (CFException)
{
	TRBEGIN("CFArchive::InitRootArchiveAndEnvironment");
	TRACE(<< "pathname=" << pathname);
	TRACE(<< "emuImageDir=" << emuImageDir);

	m_state = ROOT; //root archive will always have this state

#ifndef _MSC_VER
	//set the umask which will lead to a 755 mode for created dirs
	::umask(0022);
#endif

	ReadSettings();

	//first work in temp dir - will be changed later to dir in tree
	ASSERT(!ms_tmpdir.empty());
	m_workdir = ms_tmpdir;

	SetPathComponents(pathname);	
	CreateAndSetWorkdir();
	
	ms_emuArchiveBasename = pathname;

	//create the misc dir where temporary files are placed
	ASSERT((m_dirname.size() == 0)
		|| (m_dirname[m_dirname.size() - 1] == '/'));
	ms_miscDir = m_dirname + "misc" + "/";
#ifdef _MSC_VER
	if (::mkdir(ms_miscDir.c_str()) != 0) //create directory
#else
	if (::mkdir(ms_miscDir.c_str(), 0777) != 0) //create directory
#endif        
	{
		exc = CFException(CFException::CREATE_DIRECTORY_FAILED,
			__LINE__, __FILE__, ms_miscDir, "", errno);
		throw exc;
	}
	//create a dummy file because cleanup will explicitly
	//delete ms_miscDir. If it is emtpty it will potentially be deleted
	//in cleanups of lower level archives (lower than root). Then an
	//exception is raised
	ofstream flag((ms_miscDir + "flag").c_str(), ios::out | ios::binary);
	flag.close();

	if (emuImageDir.empty())
	{
		ASSERT((m_dirname.size() == 0)
			|| (m_dirname[m_dirname.size() - 1] == '/'));
		//create a subdirectory for the images
		//they may not appear topmost as they would be processed
		//like all other original archives
		ms_emuImageDir = m_dirname + "images" + "/";
#ifdef _MSC_VER
		if (::mkdir(ms_emuImageDir.c_str()) != 0) //create directory
#else
		if (::mkdir(ms_emuImageDir.c_str(), 0777) != 0) //create directory
#endif        
		{
			exc = CFException(CFException::CREATE_DIRECTORY_FAILED,
				__LINE__, __FILE__, ms_emuImageDir, "", errno);
			throw exc;
		}
	}
	else
	{
		ms_emuImageDir = emuImageDir;
		::BuildAbsoluteDirectoryPath(ms_emuImageDir);
		//make sure that there is no trailing slash
		ASSERT(ms_emuImageDir[ms_emuImageDir.size() - 1] != '/');
		ms_emuImageDir += '/';
		//if emu image directory is explicitly given
		//switch off deleting of this directory
		ms_bEmuImageCleanup = false;
	}
	TRreturn;
}


void CFArchive::CleanUpRootArchiveFilesystem(void) const throw (CFException)
{
	ASSERT(m_state == ROOT); //only allowed from root archive
	for (vector<CFArchive *>::size_type i = 0; i < m_files.size(); i++)
	{
		m_files[i]->CleanUpFilesystem();
	}
	CleanUpDirectory(ms_miscDir);
	//for the mother of all archives delete the emu image dir
	//only if this is wanted (no explicit image dir at command line)
	if (ms_bEmuImageCleanup)
	{
		CleanUpDirectory(ms_emuImageDir);
	}
	CleanUpDirectory(m_workdir); //directory of this archive
	//now we test if whole temporary directory structure is deleted
	struct stat buf;
	if (::stat(m_workdir.c_str(), &buf) != -1)
	{
		//top directory still there - give warning
		exc = CFException(CFException::FILE_REMOVAL_FAILED,
			__LINE__, __FILE__, m_workdir);		
		exc.WriteOutExceptionWithWarningHeader(cerr);
	}
}


void CFArchive::CleanUpFilesystem(void) const throw (CFException)
{
	for (vector<CFArchive *>::size_type i = 0; i < m_files.size(); i++)
	{
		m_files[i]->CleanUpFilesystem();
	}
	CleanUpDirectory(m_workdir); //directory of this archive
}


void CFArchive::CleanUpDirectory(string dirName) throw (CFException)
{
	TRBEGIN("CFArchive::CleanUpDirectory");
	TRACE(<< "dirName=" << dirName);
    DIR *pDir;
    struct dirent *pEntry;
	CFException excWarning;

	//this may happen if exception occurs while constructing a CFArchive
	//directories are set while constructing
	if (dirName.empty())
	{
		TRreturn;
	}
	//remove trailing / - will not affect actual parameter (call by val)
	if ((dirName.size() > 0)
		&& (dirName[dirName.size() - 1] == '/'))
	{
		dirName.resize(dirName.size() - 1);
	}
	ASSERT((dirName != "/") && (dirName != "c:\\") && (dirName != "\\"));
	TRACE(<< "Cleaning directory " << dirName);	
	if ((pDir = ::opendir(dirName.c_str())) == NULL)
		if (errno != ENOENT)
		{
			exc = CFException(CFException::OPEN_DIRECTORY_FAILED,
				__LINE__, __FILE__, dirName, "", errno);
			throw exc;
		}
		else
			TRreturn; //already cleaned up

	try {
	struct stat buf;
	
	//loop over each file in directory
	while ((pEntry = ::readdir(pDir)) != NULL)
	{
		//ignore "." and ".." for cleanup which are always found
		//by readdir - corrected RQID 9
		if (::strcmp(pEntry->d_name, ".") && ::strcmp(pEntry->d_name, ".."))
		{
			if (::stat((dirName + "/" + pEntry->d_name).c_str(), &buf) != 0)
			{
				exc = CFException(CFException::FILE_STAT_FAILED,
					__LINE__, __FILE__, dirName + "/" + pEntry->d_name
					, "", errno);
				throw exc;
			}
			//only regular files are to be removed
			if (S_ISREG(buf.st_mode))
			{
#ifdef _MSC_VER
				//only writable files can be removed for M$
				(void)::chmod((dirName + "/" + pEntry->d_name).c_str(), _S_IWRITE);
#endif
				if (::remove((dirName + "/" + pEntry->d_name).c_str()))
				{
					excWarning = CFException(CFException::FILE_REMOVAL_FAILED,
					__LINE__, __FILE__, dirName + "/" + pEntry->d_name, "", errno);
				}
			}
			else
			{
				//should be cleaned by previous subsequent destructors
				if (S_ISDIR(buf.st_mode))
				{
					//do not check success of dir removal
					//removal of top dir is checked later on
					(void)::rmdir((dirName + "/" + pEntry->d_name).c_str());
				}
				else
				{
					//unexpected file;
					(void)::closedir(pDir);
					exc = CFException(CFException::UNEXPECTED_FILE_TYPE,
						__LINE__, __FILE__, dirName + "/" + pEntry->d_name
						, CFException::IntToString(buf.st_mode));
					throw exc;
				}
			}
		}
	}

	} //END try
	catch (...)
	{
		(void)::closedir(pDir); //close the directory if processing fails
		throw;
	}

	if (::closedir(pDir) != 0)
	{
		exc = CFException(CFException::CLOSE_DIRECTORY_FAILED,
			__LINE__, __FILE__, dirName, "", errno);
		throw exc;
	}
	//do not check success of dir removal
	//removal of top dir is checked later on
	(void)::rmdir(dirName.c_str());
	//if we have encountered a warning write it out
	if (excWarning.GetExceptionId() != CFException::EXCEPTION_INVALID)
	{
		excWarning.WriteOutExceptionWithWarningHeader(cerr);
	}
	TRreturn;		
}

//SplitPathname may throw exception
void CFArchive::SetPathComponents(const string& pathname) throw (CFException)
{
	TRBEGIN("CFArchive::SetPathComponents");
	TRACE(<< "pathname=" << pathname);
	m_pathname = pathname;
	
	::SplitPathname(m_pathname, m_dirname, m_filename, m_extension);

	TRreturn;	
}


//creator of any archive
CFArchive *CFArchive::CreateSubClassObject(const string& path
	, const unsigned int blockSize) const throw (CFException)
{
	TRBEGIN("CFArchive::CreateSubClassObject");
	TRACE(<< "pathname=" << path << " blockSize=" << blockSize);
	CFArchive *pSubClass = NULL;
	
	//check if file is of generic archive class
	//iterate over the user defined generic archive classes
	//WE DO THIS FIRST TO HAVE A REPLACEMENT FOR EXTERNAL HELPER
	//MODE. GENERIC ARCHIVE ON BUILTIN IS PRIOR TO BUILTIN.
	for (vector<CFGenericArchive>::size_type i = 0
		; i < CFGenericArchive::ms_genericArchives.size(); i++)
	{
		if (CFGenericArchive::ms_genericArchives[i]
			.CheckFileIsOfClass(path))
		{
			//if extension matches then create an
			//archive of generic type
			//take over extraction command and extension
			//from prototype
			pSubClass = new CFGenericArchive(
				path, 
				CFGenericArchive::ms_genericArchives[i]
				.GetGenericExtractCommand(),
				CFGenericArchive::ms_genericArchives[i]
				.GetGenericExtension(),
				blockSize);
			break;
		}
	}
	if (pSubClass == NULL)
	{
		//only check for builtins after generic archives did not match
		if (CFZipC64Archive::CheckFileIsOfClass(path))
		{
			pSubClass = new CFZipC64Archive(path, blockSize);
		}
		else if (CFLhaArchive::CheckFileIsOfClass(path))
		{
			pSubClass = new CFLhaArchive(path, blockSize);
		}
		else if (CFLnxArchive::CheckFileIsOfClass(path))
		{
			pSubClass = new CFLnxArchive(path, blockSize);
		}
		else if (CFD64Archive::CheckFileIsOfClass(path))
		{
			pSubClass = new CFD64Archive(path, blockSize);
		}
		else if (CFT64Archive::CheckFileIsOfClass(path))
		{
			pSubClass = new CFT64Archive(path, blockSize);
		}
		else if (CFZipArchive::CheckFileIsOfClass(path))
		{
			pSubClass = new CFZipArchive(path, blockSize);
		}
		else if (CFGzipArchive::CheckFileIsOfClass(path))
		{
			pSubClass = new CFGzipArchive(path, blockSize);
		}
		else if (CFX64Archive::CheckFileIsOfClass(path))
		{
			pSubClass = new CFX64Archive(path, blockSize);
		}
		else if (CFPc64Archive::CheckFileIsOfClass(path))
		{
			pSubClass = new CFPc64Archive(path, blockSize);
		}
	}

	//nice to have a final file
	if (!pSubClass)
	{
		pSubClass = new CFFinalArchive(path, blockSize);
	}
	ASSERT(pSubClass != NULL);
	TRACE(<< "Created object is of class: " << typeid(*pSubClass).name());
	//son has to remember who his parent is (to traverse up)
	pSubClass->m_pParentArchive = const_cast<CFArchive *>(this);
	//working directory is copied (to build directory tree)
	pSubClass->m_workdir = m_workdir;
	//if a D64 image has been created from this archive
	//take over the request order of the emulator image
	//meanwhile there could have been many other D64 images
	//requested so it would be wrong to set the order when
	//constructing the D64 archive!
	//meaningless for other archives than emulator images!
	pSubClass->m_nRequestOrderIndex = m_nRequestOrderIndex;
	TRreturn pSubClass;
}


void CFArchive::ProcessRootArchive(void) throw (CFException)
{
	TRBEGIN("CFArchive::ProcessRootArchive");
	
	ASSERT(m_state == ROOT);
	ASSERT(!m_pParentArchive);

#ifdef NOCOPY

	for (vector<CFArchive *>::size_type i = 0; i < m_files.size(); i++)
	{
		try {
		m_files[i]->ProcessFilesInArchive();
		}
		catch (CFException& actExc)
		{
			//on level of root archive we declare input file processing as failed
			//and continue with next input file (do not throw exception)
			actExc.WriteOutExceptionWithWarningHeader(cerr);
		}
	}

	//flush potentially opened disk image with final archives of this archive
	CloseActualEmuImage();
	
	ProduceWarningsForIncompleteMultipart();

	if (ms_emuArchives.size() == 0)
	{
		exc = CFException(CFException::APPLICATION_ERROR,
			__LINE__, __FILE__, "", "");
		throw exc;
	}
#else

	bool bProcessingFailure = false;

	try {
	ProcessFilesInArchive();
	ProduceWarningsForIncompleteMultipart();
	}
	catch (CFException& actExc)
	{
		actExc.WriteOutExceptionWithWarningHeader(cerr);
		bProcessingFailure = true;
	}

	if (bProcessingFailure || (ms_emuArchives.size() == 0))
	{
		exc = CFException(CFException::APPLICATION_ERROR,
			__LINE__, __FILE__, "", "");
		throw exc;
	}
#endif
	TRreturn;
}


bool CFArchive::ProcessSingleFileOfArchive(struct dirent *pEntry)
	throw (CFException)
{
	TRBEGIN("CFArchive::ProcessSingleFileOfArchive");
	TRACE(<< "Filename: " << pEntry->d_name);
	struct stat buf;

	if (::stat((m_workdir + "/" + pEntry->d_name).c_str(), &buf) != 0)
	{
		exc = CFException(CFException::FILE_STAT_FAILED,
			__LINE__, __FILE__, m_workdir + "/" + pEntry->d_name
			, "", errno);
		throw exc;
	}
	//only regular files are to be processed
	if (S_ISREG(buf.st_mode))
	{
		string authenticName = pEntry->d_name;
		
		//calculate the blocksize of the file
		unsigned int blockSize = ((unsigned int)buf.st_size +  BLOCKSIZE_NETTO - 1) / BLOCKSIZE_NETTO;
		TRACE(<< "Processing file " << (m_workdir + "/" +
				authenticName));
		//create the right object and queue it
		CFArchive *newEntry = CreateSubClassObject(m_workdir + "/" + authenticName, blockSize);
		m_files.push_back(newEntry);
		
		//descent into archive
		//for final archives put them into D64
		newEntry->ProcessFilesInArchive();
		TRreturn true; //new archive processed
	}
	TRreturn false; //nothing done
}


void CFArchive::ProcessFilesInArchive(void) throw (CFException)
{
	TRBEGIN("CFArchive::ProcessFilesInArchive");
	if (m_state != EXTRACTED)
		Extract(); //to process files we need extracted archive

	DIR *pDir;
	struct dirent *pEntry;
		
	if ((pDir = ::opendir(m_workdir.c_str())) == NULL)
	{
		exc = CFException(CFException::OPEN_DIRECTORY_FAILED,
			__LINE__, __FILE__, m_workdir, "", errno);
		throw exc;
	}
	
	//loop over each file in directory
	while ((pEntry = ::readdir(pDir)) != NULL)
	{
		try {
		//if dir entry is an archive create archive object and process it further
		(void)ProcessSingleFileOfArchive(pEntry);
		} // END try
		catch (CFException& actExc)
		{
			if (m_state == ROOT)
			{
				//on level of root archive we declare input file processing as failed
				//and continue with next input file (do not throw exception)
				actExc.WriteOutExceptionWithWarningHeader(cerr);
			}
			else
			{
				//on any other level we clean up and go up
				(void)::closedir(pDir); //close the directory if processing fails
				CloseActualEmuImage();
				throw;
			}
		}
	}
	if (::closedir(pDir) != 0)
	{
		exc = CFException(CFException::CLOSE_DIRECTORY_FAILED,
			__LINE__, __FILE__, m_workdir, "", errno);
		throw exc;
	}
	//flush potentially opened disk image with final archives of this archive
	CloseActualEmuImage();

	TransitionTo(PROCESSED);

	if (m_files.size() == 0)
	{
		//we have not recognized a single file in archive
		//this will lead to a warning
		exc = CFException(CFException::ARCHIVE_EMPTY,
			__LINE__, __FILE__, GetPathname(), "");
		exc.WriteOutExceptionWithWarningHeader(cerr);
	}
	TRreturn;
}


void CFArchive::Extract(void) throw (CFException)
{
	//this must not happen - how should we extract an abstract archive?
	ASSERT(0);
}


void CFArchive::CloseActualEmuImage(void) throw (CFException)
{
	TRBEGIN("CFArchive::CloseActualEmuImage");

	if (m_pDiskImage)
	{
		//flush still opened disks
		m_pDiskImage->WriteDiskMemoryToFilesystem();
		//create the d64 for the emulator
		CFArchive *tmpArchive = CreateSubClassObject(
			m_pDiskImage->GetFilename());
		//add it to emulator images
		ms_emuArchives.push_back(tmpArchive);
		delete m_pDiskImage;
		m_pDiskImage = NULL;
	}
	TRreturn;
}


//this is the level scan for belonging multipart archives
bool CFArchive::TraverseAllArchivesOfLevelSearchingMultipart(
	CFArchive& candidate //the incomplete multipart archive to complete
	, unsigned int actualLevel //if 0 we are on the right level to scan
)
{
	TRBEGIN("CFArchive::TraverseAllArchivesOfLevelSearchingMultipart");
	TRACE(<< "Level is: " << actualLevel << " (0 to begin processing).");
	TRACE(<< "Top archive is: " << GetPathname());
	vector<CFArchive *>::iterator i = m_files.begin();
	while (i != m_files.end())
	{
		bool bIncrement = true;
		if (actualLevel > 0)
		{
			//step deeper to reach the layer of interest
			bool completed =
			(*i)->TraverseAllArchivesOfLevelSearchingMultipart(candidate
				, actualLevel - 1);
			//if we have completed the multipart break the recursion
			if (completed)
				TRreturn true;
		}
		else
		{
			//we are at the layer of interest
			if (&candidate != *i) //don't check the candidate itself
			{
				CFMultiPart *pMultiPartCandidate
					= dynamic_cast<CFMultiPart *>(&candidate);
				ASSERT(pMultiPartCandidate);
				ASSERT(*i != NULL);
				//1. is archive of same type as candidate?
				//2. is the multipart archive to merge incomplete?
				if ((typeid(**i) == typeid(candidate))
					&& !dynamic_cast<CFMultiPart *>(*i)->TestCompleteness())
				{
					//check if counterpart belongs to this part
					//if so set part to avail and remember pathname
					if (pMultiPartCandidate
						->CheckBelongingAndMergeIfBelongs(**i))
					{
						//now remove the previously merged archive
						CFArchive *delArchive = *i;
						i = m_files.erase(i); //from child array
						delete delArchive; //and from dynamic memory
						//i points now one element further than before erasure
						//do not increment
						bIncrement = false;
						//if we have completed the multipart indicate this
						//to end the recursion and signal the caller
						if (pMultiPartCandidate->TestCompleteness())
							TRreturn true;
					}

				}
			}
		}
		if (bIncrement)
		{
			//only increment if indicated
			i++;
		}
	}
	//multipart not yet completed
	TRreturn false;
}


//For searching m_files for a non complete multi part archive
//Necessary for use of STL algorithm find_if
class CAlgoFindIncompleteMultiPart
{
public:
	int operator() (CFArchive*& arch)
	{
		CFMultiPart const *pMultiPart;
		if (((pMultiPart = dynamic_cast<const CFMultiPart *>(arch)) != NULL)
			&& !pMultiPart->TestCompleteness())
			return true;
		return false;
	}
};


//this is called after all processing has happened
//if there are still incomplete multipart archives this means
//error
void CFArchive::ProduceWarningsForIncompleteMultipart(void)
{
	vector<CFArchive *>::iterator i =
		find_if(m_files.begin(), m_files.end()
			, CAlgoFindIncompleteMultiPart());
	while (i != m_files.end())
	{
		//argh we have found one
		//we have no further chance to complete the multipart
		exc = CFException(CFException::INCOMPLETE_MULTI_PART_ARCHIVE,
			__LINE__, __FILE__, (*i)->GetPathname(), "");
		exc.WriteOutExceptionWithWarningHeader(cerr);
		//do not scan for actual archive anymore
		i =	find_if(i + 1, m_files.end()
			, CAlgoFindIncompleteMultiPart());
	}
	//check children for incomplete multiparts
	for (i = m_files.begin(); i != m_files.end(); i++)
	{
		(*i)->ProduceWarningsForIncompleteMultipart();
	}
}


const string& CFArchive::BuildUniqueEmuImageName(void) const throw (CFException)
{
	TRBEGIN("CFArchive::BuildUniqueEmuImageName");
	static string uniqueFilename;
	
	//this vector to keep track of already produced emu image
	//names. this is necessary because a requested name does
	//NOT mean that this name already appears in filesystem
	//so we have to keep track of the names
	static vector<string> usedEmuImageNames;

	//a new emu image has been requested - increment the counter for emu images
	//remember the request order of D64 image
	//this is NOT necessarily the construction order of the D64 archive
	m_nRequestOrderIndex = ++ms_emuArchiveCount;

	if (ms_bEmuImageNameDueOriginal)
	{
		//try to keep part of the source archive in the emu image name
		TRACE(<< "Building unique emulator image name for archive: " << m_pathname);
		uniqueFilename = m_filename;
		string unused, filename, extension;
		//strip off any extension of the filename
		do
		{
			::SplitPathname(uniqueFilename, unused, filename, extension);
			uniqueFilename = filename;
		}
		while (!extension.empty());
		ASSERT(!uniqueFilename.empty());
		ASSERT(ms_emuImageDir.empty() || ms_emuImageDir[ms_emuImageDir.size() - 1] == '/');
		//make filename lowercase
		for (string::size_type i = 0; i < uniqueFilename.size(); i++)
			uniqueFilename[i] = (char)tolower(uniqueFilename[i]);
		//make filename command line conform (e.g. remove blanks)
		uniqueFilename = BuildHostConformFilename(uniqueFilename);
		//build absolute path
		uniqueFilename = ms_emuImageDir + uniqueFilename + ".d64";
		//throws exception if fails
		//check within filesystem AND the vector of already
		//used names.
		uniqueFilename = ::UniqueFileNameGenerator(uniqueFilename
			, &usedEmuImageNames);
		//we store the unique filename because it is NOT yet
		//present in filesystem but ALREADY occupied!
		usedEmuImageNames.push_back(uniqueFilename);
	}
	else
	{
		//just count the emu images from target000 - target<n>
		strstream imageName;
		imageName << ms_emuImageDir << ms_emuArchiveBasename << setw(3) << setfill('0')
			<< ms_emuArchiveCount << ".d64" << '\0';
		uniqueFilename = imageName.str();
	}
	TRACE(<< "Unique filename is: " << uniqueFilename);
	TRreturn uniqueFilename;
}


string& CFArchive::NameAndCreateTmpDir(void) throw (CFException)
{
	TRBEGIN("CFArchive::NameAndCreateTmpDir");
	static string resultString;

	resultString = m_workdir + "/" + m_filename + "XXXXXX";
	char *tmp = new char[resultString.size() + 1]; //one for \0

	ASSERT(tmp != NULL);
	::strcpy(tmp, resultString.c_str());
	if (::mktemp(tmp) == NULL) //make unique directory name
	{
		delete [] tmp;
		exc = CFException(CFException::UNIQUE_NAME_CREATION_FAILED,
			__LINE__, __FILE__, resultString, "", errno);
		throw exc;
	}	
	resultString = (string)tmp;

	delete [] tmp;

#ifdef _MSC_VER
	if (::mkdir(resultString.c_str()) != 0) //create directory
#else
	if (::mkdir(resultString.c_str(), 0777) != 0) //create directory
#endif        
	{
		exc = CFException(CFException::CREATE_DIRECTORY_FAILED,
			__LINE__, __FILE__, resultString, "", errno);
		throw exc;
	}

	TRreturn resultString;
}


void CFArchive::CreateAndSetWorkdir(void) throw (CFException)
{
	TRBEGIN("CFArchive::CreateAndSetTmpDir");
	m_workdir = NameAndCreateTmpDir();

	m_dirname = m_workdir + "/";
	TRreturn;
}


void CFArchive::CopyThisArchiveToDirectory(const string& directory) throw (CFException)
{
	TRBEGIN("CFArchive::CopyThisArchiveToDirectory");
	TRACE(<< "directory " << directory);

	//directory must have trailing /
	ASSERT(!directory.empty() && directory[directory.length() - 1] == '/')
	string outName = directory + m_filename + m_extension;
	string inName = m_dirname + m_filename + m_extension;
	//first check if source is a regular file
	struct stat buf;
	if (::stat(inName.c_str(), &buf) != 0)
	{
		//no exception for "file not there" - open will fail
		if (errno != ENOENT)
		{
			exc = CFException(CFException::FILE_STAT_FAILED,
				__LINE__, __FILE__, inName, "", errno);
			throw exc;
		}
	}
	else
	{
		//only regular files are to be copied
		if (!S_ISREG(buf.st_mode))
		{
			exc = CFException(CFException::UNEXPECTED_FILE_TYPE,
				__LINE__, __FILE__, inName, CFException::IntToString(buf.st_mode));
			throw exc;
		}
	}

	//copy source archive to temporary archive
	fstream in(inName.c_str(), ios::in | ios::binary);
	if (!in)
	{
		exc = CFException(CFException::FILE_OPEN_FAILED,
			__LINE__, __FILE__, inName, "reading");
		throw exc;
	}
	
	fstream out(outName.c_str(), ios::out | ios::binary);
	if (!out)
	{
		exc = CFException(CFException::FILE_OPEN_FAILED,
			__LINE__, __FILE__, outName, "write from scratch");
		throw exc;
	}	
	char ch;

	while (in.get(ch))
	{
		(void)out.put(ch);
	}

	if (in.bad())
	{
		exc = CFException(CFException::FILE_OPERATION_FAILED,
			__LINE__, __FILE__, inName, "reading");
		throw exc;
	}
	if (!out)
	{
		exc = CFException(CFException::FILE_OPERATION_FAILED,
			__LINE__, __FILE__, outName, "writing");
		throw exc;
	}
	TRreturn;
}


const CFArchive& CFArchive::GetFirstEmuImageArchive(void)
{
	TRBEGIN("CFArchive::GetFirstEmuImageArchive");
	ASSERT(!ms_emuArchives.empty());
	//to determine the first emulator image requested while processing
	//we have to execute a search. the queuing order IS NOT necessarily
	//the requesting order of emulator images. Example a.zip contains:
	//a.prg
	//b.lnx
	//for a.prg an emu image is requested (but not yet created)
	//processing finds b.lnx and creates an emu image of it which
	//is instantly requested, created and queued (AT FIRST!).
	//emu image for a.prg is still pending and queued AT LAST!
	//so I have introduced an counter for image requests which
	//actually stores the request order which matters!
	vector<CFArchive *>::size_type  nFirstArchiveIndex = 0;
	for (vector<CFArchive *>::size_type i = 1; i < ms_emuArchives.size(); i++)
	{
		if (ms_emuArchives[i]->m_nRequestOrderIndex
			< ms_emuArchives[nFirstArchiveIndex]->m_nRequestOrderIndex)
		{
			nFirstArchiveIndex = i;
		}
	}
	TRreturn *ms_emuArchives[nFirstArchiveIndex];
}


CFArchive::tArchiveState CFArchive::TransitionTo
	(const tArchiveState newState)
{
	//mother of all archives is stamped as ROOT
	//never change this state
	if (m_state != ROOT)
	{
		m_state = newState;
	}
	return m_state;
}


#ifdef _MSC_VER
//Replace all slashes by backslashes (sigh)
string CFArchive::ConvertFilenameUnixToWin(const string& filename)
{
	string retString = filename;
	
	for (string::size_type i = 0; i < retString.size(); i++)
	{
		if (retString[i] == '/')
		{
			retString[i] = '\\';
		}
	}
	return retString;
}


//Replace all slashes by backslashes (sigh)
//and transform the long filename into a short one (DOS)
string CFArchive::ConvertFilenameUnixToWinShort(const string& filename)
	throw (CFException)
{
	string retString = ConvertFilenameUnixToWin(filename);
	
	char *shortName = static_cast<char *>(calloc(retString.size() + 1, 1));
	if (shortName == NULL)
	{
		exc = CFException(CFException::MEMORY_ALLOCATION_FAILED,
			__LINE__, __FILE__, CFException::IntToString((int)retString.size() + 1), "");
		throw exc;
	}
	if (GetShortPathName(retString.c_str(), shortName, retString.size() + 1) == 0)
	{
		free(shortName);
		exc = CFException(CFException::FILENAME_CONVERSION_LONG_SHORT_FAILED,
			__LINE__, __FILE__, retString, "");
		throw exc;
	}
	retString = shortName;
	free(shortName);
	return retString;
}
#endif


#ifdef _MSC_VER
//For UNIX commands can be concatenated in one command line in various
//ways. The most popular would be ; . Windows does not support this
//concatenation. So we have to split the command and put each single
//command into a seperate line of a batchfile and execute the batchfile
//big sigh!
int CFArchive::BuildBatchFileAndExecuteSystem(const string& command)
	throw (CFException)
{
	ASSERT(CFArchive::ms_miscDir.empty()
		|| CFArchive::ms_miscDir[CFArchive::ms_miscDir.size() - 1] == '/');

	string batchFileName = UniqueFileNameGenerator(CFArchive::ms_miscDir + "batchfile.bat");

	//strange enough newlines were not translated into \r\n
	//when opening in textmode, so I am using binary
	ofstream batch(batchFileName.c_str(), ios::out | ios::binary);
	if (!batch)
	{
		exc = CFException(CFException::FILE_OPEN_FAILED,
			__LINE__, __FILE__, batchFileName, "write from scratch");
		throw exc;
	}
	
	istrstream commandStream(command.c_str());
	//as long as there are commands in commandStream
	while (!commandStream.eof() && commandStream)
	{
		string singleCommand;
		//get a command seperated by ; or eof
		(void)getline(commandStream, singleCommand, ';');
		batch << singleCommand << "\r\n";
	}
	if (!commandStream.eof() || !batch)
	{
		exc = CFException(CFException::FILE_OPERATION_FAILED,
			__LINE__, __FILE__, batchFileName, "writing");
		throw exc;
	}
	batch.close();

	//Call the batchfile previously built
	//M$ will return 0 if command is not found - no workaround for this error
	int ret = system(CFArchive::ConvertFilenameUnixToWin(batchFileName).c_str());

	//now remove the batchfile (only if indicated)
	if (ms_bCleanup && ::remove(batchFileName.c_str()))
	{
		exc = CFException(CFException::FILE_REMOVAL_FAILED,
			__LINE__, __FILE__, batchFileName, "", errno);
		throw exc;
	}

	return ret;
}
#endif

//--------------------------------------------------------------
//Provide a system call that redirects output of call
//--------------------------------------------------------------
//TODO: make this C++ like
//Replace any C call with C++ pendants
int CFArchive::SystemRedirectOutput(const char *command)
	throw (CFException)
{
#ifdef _DEBUG

	//in case TRACE is on, print out the system output
	if (TESTTRACEON)
	{
#ifdef _MSC_VER
		return BuildBatchFileAndExecuteSystem(command);
#else
		return ::system(command);
#endif
	}

#endif
	
	//Dumping stdout & stderr of a command into a temporary file
	ASSERT(ms_miscDir.empty()
		|| ms_miscDir[CFArchive::ms_miscDir.size() - 1] == '/');
	static string filename = "";
	if (filename.empty())
	{
		//create redirection filename only once
		//append to it every time this function is called
		filename = ms_miscDir + "redirectedXXXXXX";

		char *tmpFilename = new char[filename.size() + 1];
		ASSERT(tmpFilename != NULL);
		::strcpy(tmpFilename, filename.c_str());
		if (::mktemp(tmpFilename) == NULL)
		{
			delete [] tmpFilename;
			exc = CFException(CFException::UNIQUE_NAME_CREATION_FAILED,
				__LINE__, __FILE__, filename, "", errno);
			throw exc;
		}
		
		filename = (string)tmpFilename;
		delete [] tmpFilename;
	}

	//get the handles of stdout and stderr
	int handleStdout = fileno(stdout), handleStderr = fileno(stderr);
	ASSERT((handleStdout == 1) && (handleStderr == 2));

	int oldStdout = ::dup(handleStdout);
	if (oldStdout < 0)
	{
		exc = CFException(CFException::FILE_DUP_FAILED,
			__LINE__, __FILE__, CFException::IntToString(handleStdout)
			, "", errno);
		throw exc;
	}
	int oldStderr = ::dup(handleStderr);
	if (oldStderr < 0)
	{
		(void)::close(oldStdout);
		exc = CFException(CFException::FILE_DUP_FAILED,
			__LINE__, __FILE__, CFException::IntToString(handleStderr)
			, "", errno);
		throw exc;
	}
		
	FILE *newOut = ::fopen(filename.c_str(), "ab");
	if (newOut == NULL)
	{
		(void)::close(oldStdout);
		(void)::close(oldStderr);
		exc = CFException(CFException::FILE_OPEN_FAILED,
			__LINE__, __FILE__, filename, "append writing", errno);
		throw exc;
	}
	      
	if (dup2(fileno(newOut), handleStdout) < 0)
	{
		(void)::fclose(newOut);
		(void)::close(oldStdout);
		(void)::close(oldStderr);
		exc = CFException(CFException::FILE_DUP2_FAILED,
			__LINE__, __FILE__, CFException::IntToString(fileno(newOut))
			, CFException::IntToString(handleStdout), errno);
		throw exc;
	}
	if (::dup2(fileno(newOut), handleStderr) < 0)
	{
		(void)::fclose(newOut);
		(void)::dup2(oldStdout, handleStdout);
		(void)::close(oldStdout);
		(void)::close(oldStderr);
		exc = CFException(CFException::FILE_DUP2_FAILED,
			__LINE__, __FILE__, CFException::IntToString(fileno(newOut))
			, CFException::IntToString(handleStderr), errno);
		throw exc;
	}

	int ret = -1;
	errno = 0;
	try {
#ifdef _MSC_VER
	ret = BuildBatchFileAndExecuteSystem(command);
#else
	ret = ::system(command);
#endif
	}
	catch (...)
	{
		//make sure that in case of exception redirection is taken back
		(void)::fclose(newOut);
		(void)::dup2(oldStdout, handleStdout);
		(void)::dup2(oldStderr, handleStderr);
		(void)::close(oldStdout);
		(void)::close(oldStderr);
		throw;
	}

	//we asume that the following calls go well and so errno is not
	//affected and reflects system()
	(void)::fclose(newOut);
	(void)::dup2(oldStdout, handleStdout);
	(void)::dup2(oldStderr, handleStderr);
	(void)::close(oldStdout);
	(void)::close(oldStderr);
	
	return ret;
}


#ifdef _MSC_VER
//--------------------------------------------------------------
//Provide UNIX Scan Directory functions - what a hack
//--------------------------------------------------------------

//MS and POSIX - no!
//use the strange FindFirstFile FindNextFile functions to traverse directory
//this is a quick and dirty implementation of opendir, readdir, closedir

static WIN32_FIND_DATA dirFileData;

static DIR *opendir(const char *name)
{
	//allocate a new directory handle
	DIR *newDIR = new DIR;
	ASSERT(newDIR != NULL);

	char pwd[MAX_PATH];
	if (!GetCurrentDirectory(MAX_PATH, pwd))
	{
		errno = (int)GetLastError();
		delete newDIR;
		return NULL;
	}

	//set the directory to scan
	if (!SetCurrentDirectory(name))
	{
		errno = (int)GetLastError();
		delete newDIR;
		return NULL;
	}

	*newDIR = FindFirstFile("*.*", &dirFileData);
	if (*newDIR == INVALID_HANDLE_VALUE)
	{
		errno = (int)GetLastError(); //set system error as opendir would do for UNIX
		delete newDIR;
		return NULL;
	}

	//set back the current directory
	if (!SetCurrentDirectory(pwd))
	{
		errno = (int)GetLastError(); //set system error as opendir would do for UNIX
		delete newDIR;
		return NULL;
	}

	return newDIR;
}

static int closedir(DIR *dir)
{
	if (*dir == NULL)
	{
		errno = 0; //who knows what went wrong
		return 1;
	}
	if (FindClose(*dir))
	{
		delete dir;
		return 0;
	}
	*dir = NULL;
	errno = (int)GetLastError(); //set system error as closedir would do for UNIX
	return 1;
}


//TODO: "." and ".." are not read - this may cause problems for
//      other programs than TargetD64
static struct dirent *readdir(DIR *dir)
{
	static dirent puf;

	if (*dir == NULL)
	{
		errno = 0; //who knows what went wrong
		return NULL;
	}

	//this relies on the fact that . and .. come first
	//. has been found by opendir
	if (!FindNextFile(*dir, &dirFileData))
	{
		if (GetLastError() != ERROR_NO_MORE_FILES)
		{
			//closedir will give an error for this one
			(void)FindClose(*dir);
			*dir = NULL;
		}
		return NULL;
	}
	//read over . and ..
	if (!strcmp(dirFileData.cFileName, ".."))
	{
		if (!readdir(dir))
			return NULL;
	}

	strcpy(puf.d_name, dirFileData.cFileName);

	return &puf;
}

#endif
