#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include "module.h"

Module::Module(void)
  : error_message(0),
    PatternData(0),
    samples(0),
    in(0),
    rows(0),
    order(0),
    name(0)
{
}

Module::~Module(void)
{
  dump();
}

bool Module::is_module(const char *filename)
{
  if (strstr(filename, ".mod") ||
      strstr(filename, ".xm") ||
      strstr(filename, ".it") ||
      strstr(filename, ".s3m"))
    return true;

  return false; 
}

int Module::dump(void)
{
  if (name)
    {
      free(name);
      name = 0;
    }
  if (order)
    {
      free(order);
      order = 0;
    }
  if (rows)
    {
      free(rows); 
      rows = 0;
    }
  if (PatternData)
    {
      for (int p = NumPatterns; p; p--)
	for(int ch = channels; ch; ch--)
	  if (PatternData[p-1][ch-1])
	    free(PatternData[p-1][ch-1]);
      free(PatternData);
      PatternData = 0; 
    }
  if (in)
    {
      for (int i = NumInst; i; i--)
	in[i-1]->dump();
      free(in);
      in = 0;
    }
  else 
    {
      // modules with instruments map their instruments samples
      // to mod->Samples,make sure we don't free them twice!
      if (samples)
	{
	  for (int i = NumSamples; i; i--)
	    if (*(samples + i-1))
	      samples[i-1]->dump();
	  free(samples);
	}
    }
  return 1;
}

/*****************************************************************************\
 .                  Module loaders - ;mod, s3m, xm, it ?                     .
\*****************************************************************************/
// generic loader
int Module::load(const char *filename)
{
  if       (strstr(filename, ".mod")) 
    return loadMOD(filename);
  else if  (strstr(filename, ".xm")) 
    return  loadXM(filename);
  else if  (strstr(filename, ".it")) 
    return loadIT (filename);
  else if  (strstr(filename, ".s3m")) 
    return loadS3M(filename);

  error_message = "load error";
  return 0;  
}
///////////////////////////////////////////////////////////////////////////////
//                       utilities for loading                               //
///////////////////////////////////////////////////////////////////////////////
uword finetune_to_C2SPD(const ubyte finetune)
{
  switch (finetune)
    {
    case  0: return 8363;
    case  1: return 8413;
    case  2: return 8463;
    case  3: return 8529;
    case  4: return 8581;
    case  5: return 8651;
    case  6: return 8723;
    case  7: return 8757;
    case  8: return 7895;
    case  9: return 7941;
    case 10: return 7985;
    case 11: return 8046;
    case 12: return 8107;
    case 13: return 8169;
    case 14: return 8232;
    case 15: return 8280;
    default: return 8363;
    }
}

// search the period table for period
byte period_to_note(const word period)
{
  int lo, hi,mid;
  //  if (!period)
  //    return 0;
  lo = -1;
  hi = 128;  
  while (hi!=lo+1)
    {
      mid = (hi + lo) >> 1;
      if (period_table[mid] == period)
	return mid+1;
      if (period_table[mid] < period)
	hi = mid;
      else
	lo = mid;
    }
    return 0;
}

byte convert_s3mfx(const byte fx, byte *param)
{
  const byte x = *param >> 4;
  const byte y = *param & 0xf;
  switch (fx)
    {
    case 0x00: // effect param with no effect
      return 0xff;
    case 0x01: // A (set speed)
      return SPEED_SET;
    case 0x02: // B
      return PATTERN_JUMP;
    case 0x03:// C
      return PATTERN_BREAK;
    case 0x04:// D -- needs more -- normal or fine ? 
      if ((y == 0xf) && (x > 0)) // this includes DFF
	{
	  *param = x;
	  return FINE_VOL_SLIDE_UP;
	}
      else if ((x == 0xf) && (y > 0))
	{
	  *param = y;
	  return FINE_VOL_SLIDE_DN;
	}
      return VOLUME_SLIDE;    
    case 0x05:// E -- needs more -- 
      if (y == 0xe) // extra fine portamento down
	return XTR_FINE_PORT_DN;
      else if (y == 0xf) // fine portamento down
	return FINE_PORT_DN;
      return PORT_DOWN;
    case 0x06:// F
      if (y == 0xe) // extra fine portamento down
	return XTR_FINE_PORT_UP;
      else if (y == 0xf) // fine portamento down
	return FINE_PORT_UP;
      return PORT_UP;
    case 0x07:// G
      return PORT_TONE;
    case 0x08:// H
      return VIBRATO;
    case 0x09:// I
      return TREMOR;
    case 0x0a:// J
      return ARPEGGIO;
    case 0x0b:// K
      return VIBRATO_VOL_SLIDE;
    case 0x0c:// L
      return PORT_VOL_SLIDE;
    case 0x0f:// O
      return SAMPLE_OFFSET;
    case 0x11:// Q
      return RETRIG_VOL_SLIDE;
    case 0x12:// R
      return TREMOLO;
    case 0x13:// S -- based on param
      switch (*param >> 4)
	{
	case 0x0:
	  return FILTER_SET;
	case 0x1:
	  return GLISSANDO;
	case 0x2:
	  return FINETUNE_SET;
	case 0x3:
	  return VIBRATO_WAVEFORM;
	case 0x4:
	  return TREMOLO_WAVEFORM;
	case 0xA:
	  return STEREO_CONTROL;
	case 0xB:
	  return PATTERN_LOOP;
	case 0xC:
	  return NOTE_CUT;
	case 0xD:
	  return NOTE_DELAY;
	case 0xE:
	  return PATTERN_DELAY;
	case 0xF:
	  return FUNK_IT;	  
	}
      return 0xff;
    case 0x14:// T
      return SPEED_SET;      
    case 0x15:// U      
      return FINE_VIBRATO;
    case 0x16:// V      
      return GLOBAL_VOLUME;
    case 0x18: // X
      return PAN;
    }
#ifdef DEBUG
  printf("unknown effect %2X%2X\n", fx,*param);
#endif
  return 0xff;//something went wrong
}

byte convert_xmfx(const byte fx, byte *param)
{
  switch (fx)
    {
    case 0x00: 
    case 0x01: 
    case 0x02: 
    case 0x03: 
    case 0x04: 
    case 0x05: 
    case 0x06: 
    case 0x07: 
    case 0x08: 
    case 0x09: 
    case 0x0a: 
    case 0x0b: 
    case 0x0c: 
    case 0x0d: 
    case 0x0f:
      return fx;       
    case 0x0e: 
      switch (*param >> 4)
	{
	case 0x0:
	  return FILTER_SET;
	case 0x1:
	  return FINE_PORT_UP;
	case 0x2:
	  return FINE_PORT_DN;
	case 0x3:
	  return GLISSANDO;
	case 0x4:
	  return VIBRATO_WAVEFORM;
	case 0x5:
	  return FINETUNE_SET;
	case 0x6:
	  return PATTERN_LOOP;
	case 0x7:	  
	  return TREMOLO_WAVEFORM;
	case 0x9:
	  return RETRIG_NOTE;
	case 0xA:
	  return FINE_VOL_SLIDE_UP;
	case 0xB:
	  return FINE_VOL_SLIDE_DN;
	case 0xC:
	  return NOTE_CUT;
	case 0xD:
	  return NOTE_DELAY;
	case 0xE:
	  return PATTERN_DELAY;
	case 0xF:
	  return FUNK_IT;	  
	}
    case 0x10: // G
      return GLOBAL_VOLUME;
    case 0x11: // H
      return GLOBAL_VOLUME_SLIDE;
    case 0x14: // K
      return KEY_OFF;
    case 0x15: // L
      return SET_ENVELOPE_POSITION;
    case 0x19: // P
      return PANNING_SLIDE;
    case 0x1b: // R
      return RETRIG_VOL_SLIDE;
    case 0x1d: // T
      return TREMOR;
    case 0x21: // X1, X2
      {
	const byte x = *param >> 4;
	if (x == 1)
	  return XTR_FINE_PORT_UP;
	else if (x == 2)
	  return XTR_FINE_PORT_DN;
      }
    }
  return 0xff;
}


//\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\
// specific loaders
//\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\

int Module::loadMOD(const char *filename) 
{ 
  udword chunkid; 
  ubyte *TempPD = NULL; // used to store the pattern data before converting 
  //     ubyte Order[128]; // Pattern order 
  ModSampleHeader inst[31]; 
  FILE *fp = fopen(filename, "rb"); 
     
  if (!fp) 
    { 
      error_message = "No Such File";
      return 0;
    } 

  /* CHECK MAGIC ID */ 
  if (fseek(fp, 1080 ,SEEK_SET)) 
    { 
      error_message = "Seek error";
      return 0;
    } 

  if (fread(&chunkid, sizeof(char), 4, fp) != 4) 
    { 
      error_message = "read error";
      return 0;
    } 

  /* CHECK ID  & Set Channels */ 
  if (!strncmp((char *)&chunkid, "M.K.", 4)) 
    { 
      NumSamples = 31;
      channels = 4;
    } 
  else if (!strncmp((char *)&chunkid, "M!K!", 4)) 
    { 
      NumSamples = 31;
      channels = 4;
    } 
  else if (!strncmp((char *)&chunkid, "6CHN", 4)) 
    { 
      NumSamples = 31;
      channels = 6;
    } 
  else if (!strncmp((char *)&chunkid, "8CHN", 4)) 
    { 
      NumSamples = 31;
      channels = 8;
    } 
  else 
    {
      error_message = "not a MOD";
      return 0;
    } 

  // reset values of module
  dump();
  bpm     = 125;
  speed   = 6;
  NumInst = 0;
  in      = 0; // no instruments in mod
  period_shift  = 2;
  global_volume = 64;
  
#ifdef DEBUG   
  printf("Magic : [%.4s] 0x%lX\n", (char *)&chunkid, chunkid); 
#endif 
  /* REWIND TO START */ 
  if (fseek(fp, 0, SEEK_SET)) 
    { 
      error_message = "seek error"; 
      return 0;
    } 

  if ((name = (char *)malloc(21))==0)
    {
      error_message = "malloc error"; 
      return 0;
    }

  if (fread(name, sizeof(char), 20,fp) != 20) 
    { 
      error_message = "read error"; 
      return 0;
    } 

  name[21] = 0; // make it a NULL terminated string
#ifdef DEBUG 
  printf("Name : %.20s\n", name);
#endif 
  //////////////////////////////////////////////////////////////////////////  
  //                     READ IN SAMPLE HEADERS 
  /////////////////////////////////////////////////////////////////////////
  // malloc space for pointers to sample structures
  //  if ((module->Samples = (Sample **)malloc(sizeof(Sample *))) == 0)     {
  if ((samples=(Sample **)calloc(NumSamples,sizeof(Sample *)))==0) 
    {
      perror("malloc SAMPLES *");
      error_message = "malloc error";
      return 0;
    }
  
  for (int i = 0; i < NumSamples; i++) 
    { 
      if (fread(&inst[i], sizeof(ModSampleHeader), 1, fp)!=1)
	{
	  error_message = "read error";
	  return 0;
	}

      if ((samples[i] = (Sample *)calloc(1,sizeof(Sample))) == 0) 
	{
	  perror("malloc Sample");
	  error_message = "malloc error";
	  return 0;
	}

      // I DON'T KNOW WHY I NEED TO MULTIPLY BY 2 -- IT JUST WORKS !
      samples[i]->length     = B_ENDIAN_16(inst[i].Length)*2;
      samples[i]->loop_start = B_ENDIAN_16(inst[i].LoopStart)*2; 
      if (B_ENDIAN_16(inst[i].LoopLength) > 1)
	{
	  samples[i]->loop_end  = B_ENDIAN_16(inst[i].LoopLength)*2 + 
	    samples[i]->loop_start;
	  samples[i]->flags |=Sample::SMP_LOOP; // set loop flag
	} 
      else
	samples[i]->loop_end  = 0;

      samples[i]->volume     = inst[i].Volume;
      samples[i]->finetune   = (byte)inst[i].Finetune;
      samples[i]->frequency  = finetune_to_C2SPD(inst[i].Finetune); 
      samples[i]->flags     &= ~Sample::SMP_16BIT; /* 8bit only in mods */

      if ((samples[i]->name  = (char *)malloc(22))==0)
	{
	  error_message = "malloc error";
	  return 0;
	}

      if (!strncpy(samples[i]->name, inst[i].Name, 22))
	{
	  error_message = "strncpy error";
	  return 0;
	}

#if 0
      printf(" [%.2d]", i);
      print_nice(samples[i]->name,22); 
      printf("\n");
      if (samples[i]->length > 1) 
	{ 
	  printf("   Length     %ld\n", samples[i]->length); 
	  printf("   FineTune   %hd\n", samples[i]->finetune); 
	  printf("   Volume     %hd\n", samples[i]->volume); 
	  printf("   LoopStart  %ld\n", samples[i]->loop_start); 
	  printf("   LoopEnd    %ld\n", samples[i]->loop_end); 
	} 
#endif 
    } 
  fread(&SongLength, 1, 1, fp); 
  fread(&restart, 1, 1, fp); 
  NumPatterns = 0; 
  //////////////////////////////////////////////////////////////////////////  
  //                           READ IN PATTERN ORDER 
  //////////////////////////////////////////////////////////////////////////  

#ifdef DEBUG  
  printf("Pattern Orders\n"); 
#endif 

  if ((order = (ubyte *)malloc(128 * sizeof(ubyte))) == 0)
    {
      perror("malloc");
      error_message = "malloc error";
      return 0;
    }

  for (int i = 0; i < 128; i++) 
    { 
      fread(&order[i], sizeof(ubyte), 1, fp); 
#ifdef DEBUG
      if ( i < SongLength) 
	printf("[%.2X : %.2X] ", i, order[i]);
#endif
      if (order[i] > NumPatterns) 
	NumPatterns = order[i];
    }

  // patterns are numbered from zero so add one the the highest pattern no.
  NumPatterns++; 
#if 0
  printf("\nNumPatterns : %d\n", NumPatterns); 
#endif

  /* SKIP MAGIC */
  fseek(fp,4,SEEK_CUR);
  //////////////////////////////////////////////////////////////////////////  
  //                        READ IN PATTERN DATA  
  //////////////////////////////////////////////////////////////////////////  
#if 0
  printf("Read in Pattern Data\n");
#endif

  /* STANDARD - 4 channels, 4 bytes per note, 64 lines */
  if ((TempPD = (ubyte *)malloc(channels * 4 * 64)) == 0)
    {
      perror("malloc");
      error_message = "malloc error";
      return 0;
    }

  /* allocate Rows */
  if ((rows = (word *)calloc(NumPatterns,sizeof(word)))==0)
    {
      error_message = "malloc error";
      return 0;
    }

  /* allocate space for pointers */
  // pattern_ptr to channel_ptr to line_ptr to note
  if ((PatternData=(Note ***)calloc(NumPatterns, sizeof(Note **)))==0)
    {
      perror("malloc");
      error_message = "malloc error";
      return 0;
    }

  for (int i = 0; i < NumPatterns; i++) 
    {
     if ((PatternData[i]=(Note **)calloc(channels,sizeof(Note *)))==0)
       {
	 perror("malloc");
	 error_message = "malloc error"; 
	 return 0;
       }
      if (fread(TempPD, channels * 4 * 64, 1, fp) != 1) 
	{	  
	  fclose(fp);
	  perror("fread");
	  error_message = "read error";
	  return 0;
	}
      // all patterns are 64 rows long
      rows[i] = 64;
      // Need to convert TempPD to Note format and put in module->PatternData
      // i for pattern
      // j for channel
      // k for line no
      for (int j = 0; j < channels; j++) 
      	{ 
	  // allocate 64 lines for each channel
	  if ((PatternData[i][j]=(Note *)calloc(64,sizeof(Note)))==0)
	    {
	      perror("malloc");
	      error_message = "malloc error";
	      return 0;
	    }

	  for (int k = 0; k < 64; k++) 
	    {
	      ubyte *pos  = TempPD + j*4 + k*4*channels;
	      Note *n     = &PatternData[i][j][k];
	      n->number   = (*pos & 0xf0) + (*(pos + 2) >> 4);
              n->NoteNum  = period_to_note(((*(pos) & 0x0f)<<8) + *(pos + 1)); 
	      if (n->NoteNum)
		n->NoteNum -= 24;
	      // -24 st3 style notes & periods
              n->effect   = *(pos + 2) & 0x0f;
              n->effParam = *(pos + 3);	      
	      n->volume   = 0; // mods don't have a volume column
	      // convert E effects  
	      if (n->effect == 0x0E)
		n->effect = 0xE0 + (n->effParam >> 4);
            }
        }
    }  
  //////////////////////////////////////////////////////////////////////////  
  //                         READ IN SAMPLES
  //////////////////////////////////////////////////////////////////////////  
#if 0
  printf("Read in Samples\n");
#endif

#ifdef DEBUG
  printf("\n");
#endif
  /* MALLOC SPACE AND READ IN SAMPLE DATA */
  for (int i = 0; i < NumSamples; i++) 
    {
      if (samples[i]->length > 1) 
	{
	  if ((samples[i]->data = (ubyte *)calloc(samples[i]->length,1))==0) 
	    {
	      error_message = "malloc error";
	      return 0;
	    }
#if 0
	  printf("  Sample %.2d Length 0x%lX\n",i, samples[i]->length);
	  fflush(stdout);
#endif
	  if (fread(samples[i]->data,samples[i]->length,1,fp)!=1) 
	    {
	      if (ferror(fp))
		{
		  perror("fread");
		  fclose(fp);
		  error_message = "read error";
		  return 0;
		}
	    }
	  /* CONVERT TO UNSIGNED */
	  for (unsigned int j = 0; j < samples[i]->length; j++) 
	    *(samples[i]->data + j) ^= 0x80;
	} 
      else 
	{
	  if (samples[i]) 
	    {
	      samples[i]->dump();	
	      samples[i] = 0;
	    }
	}
    }
#ifdef DEBUG
  printf("NumSamples %d\n", NumSamples);
#endif
  fclose(fp);
  return 1;
}

int Module::loadS3M(const char *filename) 
{ 
  dword chunkid;
  S3MHeader Head;
  S3MSAMPLE *s3msmp = 0;
  byte *dflt_pan  = 0;
  byte *remap     = 0; // for remapping of channels
  word *para_inst = 0;
  word *para_patt = 0;
  FILE *fp = fopen(filename, "rb");
  
  if (fp == 0) 
    { 
      error_message = "No Such File";
      return 0;
    } 

  /* CHECK MAGIC ID */ 
  if (fseek(fp, 0x2c ,SEEK_SET)) 
    { 
      error_message = "Seek error"; 
      return 0;
    } 

  if (fread(&chunkid, sizeof(char), 4, fp) != 4) 
    { 
      error_message = "read error: chuckid"; 
      return 0;
    } 

  /* CHECK ID  & Set Channels */ 
  if (strncmp((char *)&chunkid, "SCRM", 4)) 
    { 
      fclose(fp);
      error_message = "not S3M!";
      return 0;
    }

  /* REWIND TO START */ 
  if (fseek(fp, 0, SEEK_SET)) 
    { 
      error_message = "seek error"; 
      return 0;
    }  

  if (fread(&Head, sizeof(S3MHeader),1,fp)!=1)
    {
      error_message = "read error: header";
      return 0;
    }

  // reset values
  dump();
  SongLength    = Head.SongLength;
  NumSamples    = Head.NumInst;
  NumPatterns   = 0; //Head.NumPatterns; // not reliable
  global_volume = Head.Global_Volume;
  bpm           = Head.BPM;
  speed         = Head.Speed;
  name          = strdup(Head.Name);
  NumInst       = 0; // s3m doesn't have instruments
  in            = 0; // no instruments in s3m
  period_shift  = 4;

#ifdef DEBUG
  printf("Name            %.28s\n", name);
  printf("Marker          %X\n",    Head.Marker);
  printf("Type            %d\n",    Head.Type);
  printf("SongLength      %d\n",    SongLength);
  printf("NumInst         %d\n",    NumSamples);
  printf("NumPatterns     %d\n",    Head.NumPatterns);
  printf("Flags           %d\n",    Head.Flags);
  if ((Head.Flags & 0x01) > 0)
    printf("st2vibrato\n");
  if ((Head.Flags & 0x02) > 0)
    printf("st2tempo\n");
  if ((Head.Flags & 0x04) > 0)
    printf("amiga slides\n");
  if ((Head.Flags & 0x08) > 0)
    printf("0 Vol optimisation\n");
  if ((Head.Flags & 0x10) > 0)
    printf("amiga limits\n");
  if ((Head.Flags & 0x20) > 0)
    printf("Enable filter/sfx\n");
  if ((Head.Flags & 0x40) > 0)
    printf("st3.00 volume slides\n");
  if ((Head.Flags & 0x80) > 0)
    printf("Custom data present\n");
  printf("Version         %X\n",     Head.Version);
  printf("Signed          %d\n",     Head.Signed);
  printf("SCRM            %.4s\n",   Head.SCRM);
  printf("Global_Volume   %d\n",     global_volume);
  printf("Speed           %d\n",     speed);
  printf("BPM             %d\n",     bpm);
  printf("Master Volume   %d\n",     Head.Master_Volume&0x7f);
  if ((Head.Master_Volume & 0x80)==0x80)
    printf("Stereo on\n");
  printf("Click_Removal   %d\n",     Head.Click_Removal);
  printf("Default_Panning %X\n",     Head.Default_Panning);
  printf("Special         %d\n",     Head.Special);
#endif  

  // now for channel settings -- next 32 bytes
  if ((remap = (byte *)malloc(32))==0)
    {
      error_message = "malloc error";
      return 0;
    }

  for (int i = 0; i < 32; i++)
    remap[i] = 0xff;

  channels = 0;

  for (int i = 0; i < 32; i++)
    { // if bit 8 is set channel is disabled
      if (Head.channels[i] != 0xff) // 0xff means channel is not used
	{
	  remap[i] = channels;
	  // set pan value for channel -- worry about that later
	  channels++;
	}
    }

#if 0
  printf("Channel Settings\n"); 
  for (int i = 0; i < channels; i++)    
    { 
      if (Head.channels[remap[i]] < 8)
	printf("L%.1X", Head.channels[remap[i]]+1);
      else if (Head.channels[remap[i]] < 16)
	printf("R%.1X", Head.channels[remap[i]]-7);
      else
	printf("  ");
    }
  printf("\n");
  // now we load the Orders
  printf("Pattern Orders\n"); 
#endif 

  if ((order = (byte *)malloc(SongLength))==0)
    {
      error_message = "malloc error";
      return 0;
    }

  if (fread(order, SongLength,1,fp)!=1)
    {
      error_message = "read error: order";
      return 0;
    }

  // need to remap the order to remove marker patterns -- 254,255
  int real_songlength = 0;
  for (int i = 0; i < SongLength; i++)
    {
      if (order[i] < 254) 
	{
	  if (order[i] > NumPatterns)
	    NumPatterns = order[i];
	  order[real_songlength++] = order[i];	  
	}
    }

  NumPatterns++;
  SongLength = real_songlength;

#ifdef DEBUG
  printf("Biggest Pattern %X\n", NumPatterns);
  for (int i = 0; i < (SongLength); i++)
    printf("[%.2X : %.2X] ", i, order[i]);
  printf("\n");
#endif

  // parapointers -- offsets in the file to instruments & patterns 
  // first need to multiply the pointer by 16
  // Instruments  -- NumInst * WORD 
  // Patterns     -- Head.NumPatterns * WORD

  if ((para_inst = (word *)malloc(Head.NumInst*2))==0)
    {
      error_message = "malloc error";
      return 0;
    }

  if (fread(para_inst,Head.NumInst*2,1,fp)!=1)
    {
      error_message = "read error: para inst";
      return 0;
    }

  if ((para_patt = (word *)malloc(Head.NumPatterns*2))==0)
    {
      error_message = "malloc error";
      return 0;
    }

  if (fread(para_patt,Head.NumPatterns*2,1,fp)!=1)
    {
      error_message = "read error: para patt";  
      return 0;
    }

#if 0
  for (int i = 0; i < Head.NumInst; i++)
    printf("%X ", para_inst[i]);
  printf("\n");

  for (int i = 0; i < Head.NumPatterns; i++)
    printf("%X ", para_patt[i]);
  printf("\n");
#endif

  // Default Pan Positions if Head.Default_Panning == 0xfc
  // 32 bytes only bottom 4bits valid
  // check if bit 8 in Master Volume was set if so then song is stereo
  if (Head.Default_Panning == 0xfc)
    {
      // read in 32 bytes of Default Panning Postions
      // use the space allocated for channels
      if ((dflt_pan = (byte *)malloc(32))==0)
	{
	  error_message = "malloc error";
	  return 0;
	}

      if (fread(dflt_pan, 32,1,fp)!=1)
	{
	  error_message = "read error: dftl pan";
	  return 0;
	}

#if 0
      printf("Default Panning Positions\n");
      for (int i = 0; i < channels; i++)
      	printf(" %.1X", dflt_pan[remap[i]] & 0xf);
      printf("\n");
#endif  
    
    }
  // Instrument Info
  // seek to parapointer * 16 for instrument and read in instrument header
  // instrument header will have another stupid file pointer to inst data
  // should I save the position in the file ?
  if ((s3msmp = (S3MSAMPLE *)malloc(Head.NumInst*sizeof(S3MSAMPLE)))==0)
    {
      error_message = "malloc error";
      return 0;
    }

  // read in instrument headers 
  for (int i = 0; i < Head.NumInst; i++)
    {
      if (para_inst[i])
	{
	  if (!fseek(fp, (para_inst[i]<<4) ,SEEK_SET)) 
	    { 
	      if (fread(&s3msmp[i],sizeof(S3MSAMPLE),1,fp)!=1)
		{
		  printf("\nerror\n%X %X\n", i, para_inst[i]);
		  error_message = "read error: s3msmp"; 
		  return 0;
		}

#if 0
	      printf("smpnum    %X\n",    i);
	      printf("filename  [%.12s]\n", s3msmp[i].filename);
	      printf("length    %lX\n",   s3msmp[i].length);
	      printf("loopstart %lX\n",   s3msmp[i].loopstart);
	      printf("loopend   %lX\n",   s3msmp[i].loopend);
	      printf("volume    %X\n",    s3msmp[i].volume);
	      printf("pack      %d\n",    s3msmp[i].pack);
	      printf("flags     %d\n",    s3msmp[i].flags);
	      printf("c2spd     %ld\n",    s3msmp[i].c2spd);
	      printf("sampname  [%.28s]\n", s3msmp[i].sampname);
	      printf("scrs      %.4s\n",  s3msmp[i].scrs);
	      printf("---\n");		 
#endif
	  
	    }
	  else
	    {
	      error_message = "seek error";
	      return 0;
	    }
	}
    }

  // malloc mod->Samples space
  if((samples=(Sample **)malloc((Head.NumInst*sizeof(Sample *))))==0) 
    {
      perror("malloc SAMPLES *");
      error_message = "malloc error";
      return 0;
    }

  for (int i = 0; i < NumSamples; i++)
    {
      if ((samples[i] = (Sample *)calloc(1,sizeof(Sample))) == 0) 
	{
	  perror("malloc Sample");
	  error_message = "malloc error";
	  return 0;
	}

      samples[i]->length = s3msmp[i].length;

      if (s3msmp[i].flags & 0x01)      
	samples[i]->flags   |=Sample::SMP_LOOP; // set loop flag      

      samples[i]->loop_start = s3msmp[i].loopstart;
      samples[i]->loop_end   = s3msmp[i].loopend;
      samples[i]->volume     = s3msmp[i].volume;
      samples[i]->finetune   = 0; // s3m doesnt have finetunes
      samples[i]->frequency  = s3msmp[i].c2spd;

      if (s3msmp[i].flags & 0x4)
	samples[i]->flags   |=Sample::SMP_16BIT;/* set 16 bit flag */
      else
	samples[i]->flags &= ~Sample::SMP_16BIT;

      samples[i]->name = strdup(s3msmp[i].sampname);
    }

  // Pattern Data
  /* allocate Rows */
  if ((rows = (uword *)calloc(NumPatterns,sizeof(uword)))==0)
    {
      error_message = "calloc error";
      return 0;
    }
  // use parapointers to seek to Pattern Data
  // need to malloc space for mods pattern data

  if ((PatternData=(Note ***)calloc(NumPatterns,sizeof(Note **)))==0)
    {
      perror("malloc");
      error_message = "malloc error";
      return 0;
    }

  for (int i = 0; i < NumPatterns; i++) 
    {
      if ((PatternData[i]=(Note **)calloc(channels,sizeof(Note *)))==0)
	{
	  perror("malloc");
	  error_message = "malloc error";
	  return 0;
	}

      // allocate 64 lines for each channel
      for (int j = 0; j < channels; j++)
	if ((PatternData[i][j] = (Note *)calloc(64,sizeof(Note)))==0)
	  {
	    perror("calloc");
	    error_message = "calloc error";
	    return 0;
	  }
    }
  
  int pat = 0;
  byte *pd = 0; // temporary space for packed pattern data

  if ((pd = (byte *)calloc(320*32,1))==0)
    {
      error_message = "malloc error";
      return 0;
    }

  for (int i = 0; i < NumPatterns; i++)
    {
      if (para_patt[i])
	{
	  pat++;
	  rows[i] = 64;
	  if (!fseek(fp, (para_patt[i]<<4) ,SEEK_SET)) 
	    {
	      uword plength;
	      // first two bytes is length of the packed pattern data
	      if (fread(&plength, 2, 1, fp)!=1)
		{
		  error_message = "read error : length";
		  return 0;
		}

	      // read in packed pattern
	      if (fread(pd,plength,1,fp)!=1)
		{
#ifdef DEBUG
		  printf("pattern %X i %X ",pat,i);
		  printf("para %X length %X\n",para_patt[i],plength);
#endif
		  error_message = "read error: pd";
		  return 0;
		}

	      // unpack pattern data
	      int row = 0;
	      int off = 0;
	      while (row < 64) 
		{
		  byte empty = 1;
		  while (off < plength) 
		    {
		      int ch;
		      int toff = off;
		      Note *n;
		      Note dummy;
		      // if byte is 0 then end of row has been reached
		      if (!pd[off]) 
			{
			  off++;
			  row++;
			  break;
			} 
		      else 
			{
			  empty = 0;
			}
	    
		      // byte & 31 is the channel
		      ch = pd[off] & 31;
		      if (remap[ch] < channels) 
			n = &(PatternData[i][remap[ch]][row]);
		      else // ignore out of range channels
			n = &dummy;
		      
		      // if byte & 32 then two more bytes follow,
		      // note value & inst no
		      if (pd[off] & 0x20) 
			{
			  // store note value & inst no
			  // note >> 4 = octave, note & 0xf = note, 
			  /// 255 = empty note, 254 = note off
			  int tmp = pd[++toff];
			  if (tmp==255)
			    n->NoteNum = 0;
			  else if (tmp == 254)
			    n->NoteNum = NOTE_OFF;
			  else 
			    n->NoteNum = ((tmp >> 4) * 12) + (tmp & 0x0f) + 1; 
			  // +25 for mod style notes & periods
			  n->number  = pd[++toff];
			}

		      // if byte & 64 then then volume byte follows
		      if (pd[off] & 0x40) // 0 means no volume set
			n->volume = pd[++toff] + 1;

		      // if byte & 128 then 2 more bytes follow 
		      // effect no & eff param
		      if (pd[off]&0x80) 
			{
			  byte fx     = pd[++toff]; 	     
			  n->effParam = pd[++toff]; 
			  // convert s3m effects to internal effects
			  n->effect   = convert_s3mfx(fx,&n->effParam);
			}		      
		      off = toff+1;
		    }
		}
	      // clear pd ?
	    }
	}
      if (pat > NumPatterns)
	break;
    }  

  // now read in sample data
  for (int i = 0; i < NumSamples; i++)
    {
      if (s3msmp[i].length) 
	{
	  const long offset = (s3msmp[i].memsegh<<16) + s3msmp[i].memsegl;
	  if (offset)
	    if (!fseek(fp,(offset<<4),SEEK_SET)) 
	      {
		if((samples[i]->data = (byte *)calloc(s3msmp[i].length, 1))==0)
		  {
		    error_message = "malloc error";
		    return 0;
		  }

		if (fread(samples[i]->data,s3msmp[i].length,1,fp)!=1) 
		  {
#ifdef DEBUG
		    printf("Sample %d Offset %ld",i,offset);
		    printf("Length %ld\n",s3msmp[i].length);
#endif
		    error_message = "read error : samples";
		    return 0;
		  }
	      }
	}
    }

  free(para_inst);
  free(para_patt);
  free(pd);
  fclose(fp);
  return 1;
} 

int Module::loadXM(const char *filename)
{
  XM_HEADER Header;
  byte *data = 0;
  FILE *fp = fopen(filename, "rb");
  
  if (fp == 0) 
    {
      error_message = "Can't open file";
      return 0;
    }

  if (fread(&Header,sizeof(XM_HEADER),1,fp)!=1)
    {
      error_message = "read error";
      return 0;
    }

  Header.name[20] = 0;
  // reset values
  dump();
  name          = strdup(Header.name);
  SongLength    = Header.SongLength;
  restart       = Header.Restart;
  channels      = Header.Channels;
  NumPatterns   = Header.Patterns; // up to 256 but who uses that many?
  NumInst       = Header.Instruments; // max 128
  NumSamples    = 0; //need to count them when processing the instruments
  speed         = Header.speed;
  bpm           = Header.bpm;
  in            = 0;
  samples       = 0;
  global_volume = 64;

  if (Header.flags & 0x01)
    {
      linear       = 1;
      period_shift = 4;	
    }
  else
    period_shift = 2;

  if ((order = (byte *)calloc(SongLength,sizeof(byte)))==0)
    {
      error_message = "calloc error";
      return 0;
    }

  for (int i = 0; i < SongLength; i++)
    order[i] = Header.order[i];
    
#ifdef DEBUG
  printf("id          %.17s\n",Header.id);  
  printf("name        %.20s\n",name);
  printf("tracker     %.20s\n",Header.tracker);
  printf("version     %X\n",Header.version);
  printf("size        0x%lX\n",Header.size);
  printf("songlength  0x%X\n",SongLength);
  printf("restart     0x%X\n",restart);
  printf("Channels    %d\n",channels);
  printf("Patterns    0x%X\n",NumPatterns);
  printf("Instruments 0x%X\n",NumInst);
  printf("flags       %d\n",Header.flags);
  printf("speed       %d\n",speed);
  printf("bpm         %d\n",bpm);
  for (int i = 0; i < SongLength;i++)
    printf("[%.2X : %.2X] ", i, order[i]);
  printf("\n");
#endif  

  if ((PatternData=(Note ***)calloc(Header.Patterns,sizeof(Note **)))==0)
    {
      perror("malloc");
      error_message = "malloc error";
      return 0;
    }

  // allocate space for rows
  if ((rows = (word *)calloc(Header.Patterns, sizeof(word)))==0)
    {
      error_message = "malloc error";
      return 0;
    }

  byte *pd = 0;
  // allocate space for packed pattern data
  // maximum of 32 channels, 256 rows and 5 bytes per note
  if ((pd = (byte *)calloc(32*256*5,sizeof(byte)))==0)
    {
      error_message = "calloc error";
      return 0;
    }

  // read in Header.Patterns patterns
  for (int p=0; p < Header.Patterns; p++)
    {
      dword head_len;
      byte type;
      word packed_len;
      byte *t = pd;

      // pattern Header consists of
      // (dword) pattern header length
      // (byte) packing type (always 0)
      // (word) Number of rows
      // (word) packed pattern data size
      // packed pattern data follows
       
      if (fread(&head_len,sizeof(dword),1,fp)!=1)
	{
	  perror("fread");
	  error_message = "read error";
	  return 0;
	}

      if (fread(&type, sizeof(byte),1,fp)!=1)
	{
	  perror("fread");
	  error_message = "read error";
	  return 0;
	}

      if (type)
	{
	  printf("pattern %d type %d\n", p, type);
	  error_message = "NOT TYPE!";
	  return 0;
	}

      if (fread(rows+p, sizeof(word),1,fp) != 1)
	{
	  perror("fread");
	  error_message = "read error";
	  return 0;
	}

      if (fread(&packed_len,sizeof(word),1,fp)!=1)
	{
	  perror("fread");
	  error_message = "read error";
	  return 0;
	}

#ifdef DEBUG
      printf("packed pattern %2X rows %2X ",p,rows[p]);
      printf("length %d\n",packed_len);
#endif

      // allocate space of for pointers to channel data
      if ((PatternData[p]=(Note **)calloc(channels,sizeof(Note *)))==0)
	{
	  perror("malloc");
	  error_message = "malloc error";
	  return 0;
	}

      if (!packed_len) // empty pattern
	{
	  rows[p] = 0;
	  continue;
	}

      // read in packed pattern
      if (fread(pd,packed_len,1,fp)!=1)
	{
	  perror("fread");
	  error_message = "read error";
	  return 0;
	}

      // allocate rows for each channel
      for (int ch = 0; ch < channels; ch++)
	{
	  if ((PatternData[p][ch] = (Note *)calloc(rows[p],sizeof(Note)))==0)
	    {
	      perror("calloc");
	      error_message = "calloc error";
	      return 0;
	    }
	}

      // interpret packed pattern
      // rather than reading the rows for each channel separately
      // each channels rows are intermixed
      for (int r = 0; r < rows[p]; r++)
	{
	  for (int ch = 0; ch < channels; ch++)
	    {
	      Note *n = &PatternData[p][ch][r];
	      byte fx = 0xff;

	      if (*t & 0x80) // is it compressed ?
		{
		  byte comp     = *(t++);
		  if (comp & 0x01)		    
		    n->NoteNum  = *(t++); 

		  if (n->NoteNum == 97)
		    n->NoteNum = NOTE_OFF;

		  if (comp & 0x02)
		    {
		      n->InstMode = 1;		      
		      n->number   = *t++; // 0 - 128
		    }

		  // xm's have some effects in the volume column
		  if (comp & 0x04)
		    {
		      n->volume   = *t++;
		      if (n->volume < 0x51)
			n->volume -= 0x0f;
		    }

		  if (comp & 0x08)
		    fx   = *t++;

		  if (comp & 0x10)
		    n->effParam = *t++;

		  if (fx != 0xff)
		    n->effect = convert_xmfx(fx, &n->effParam);
		}
	      else
		{
		  n->NoteNum  = *t++;

		  if (n->NoteNum == 97)
		    n->NoteNum = NOTE_OFF; // key off

		  n->InstMode = 1;
		  n->number   = *t++;		  
		  n->volume   = *t++;

		  if (n->volume < 0x51)
		    n->volume -= 0x0f;

		  fx          = *t++;
		  n->effParam = *t++;
		  n->effect = convert_xmfx(fx,&n->effParam);
		}	      
	    }
	}      
    }
  free(pd); 

#ifdef DEBUG
  printf("Position %lX\n",ftell(fp));
#endif

  /////////////////
  // INSTRUMENTS //
  /////////////////

  if ((in=(Instrument**)calloc(NumInst,sizeof(Instrument *)))==0)
    {
      error_message = "calloc error";
      return 0;
    }

  for (int i = 0; i < NumInst; i++)
    {  
      if ((in[i] = (Instrument *)calloc(1,sizeof(Instrument)))==0)
	{
	  error_message = "malloc error";
	  return 0;
	}

      if ((in[i]->name = (char *)calloc(22, sizeof(char)))==0)
	{
	  error_message = "malloc error";
	  return 0;
	}
    }
  // read in instruments
  for (int i = 0; i < NumInst; i++)
    {
      dword size;
      byte type; // should be 0
      XIinst xi;

      if (fread(&size,sizeof(dword),1,fp)!=1)
	{
	  perror("fread");
	  error_message = "read error";
	  return 0;
	}

      if (fread(in[i]->name, 22*sizeof(char),1,fp)!=1)
	{
	  perror("fread");
	  error_message = "read error";
	  return 0;
	}

      if (fread(&type,sizeof(byte),1,fp)!=1)
	{
	  perror("fread");
	  error_message = "read error";
	  return 0;
	}	

      if (fread(&in[i]->NumSamples, sizeof(word),1,fp)!=1)
	{
	  perror("fread");
	  error_message = "read error";
	  return 0;
	}

#ifdef DEBUG      
      printf("==Instrument %X type %X==\n",i, type);
      printf("Name           ");
      //      print_nice(in[i]->name,22);
      printf("\n");
      printf("Samples        %d\n",in[i]->NumSamples);      
#endif
      if (fread(&size,sizeof(dword),1,fp)!=1)
	{
	  perror("fread");
	  error_message = "read error";
	  return 0;
	}

      if (!in[i]->NumSamples)
	continue;

      NumSamples += in[i]->NumSamples;

      // read in Sample Header Size ( dword )
      if (fread(&xi,sizeof(XIinst),1,fp)!=1)
	{
	  perror("fread");
	  error_message = "read error";
	  return 0;
	}

      // assign values to module instruments
      // vibrato
      in[i]->vibrato.type  = xi.Vibrato_Type;
      in[i]->vibrato.sweep = xi.Vibrato_Sweep;
      in[i]->vibrato.depth = xi.Vibrato_Depth;
      in[i]->vibrato.rate  = xi.Vibrato_Rate;
      // volume
      in[i]->volume.points     = xi.Num_Vol_Points;
      in[i]->volume.sustain    = xi.Vol_Sus;
      in[i]->volume.loop_start = xi.Vol_Loop_Start;
      in[i]->volume.loop_end   = xi.Vol_Loop_End;
      in[i]->volume.type       = xi.Vol_Type;
      in[i]->volume.fadeout    = xi.Vol_Fadeout;

      for (int point = 0; point < 24; point++)
	{
	  in[i]->volume.envelope[point] = xi.Vol_Envelope[point];
	}

      // panning	  
      in[i]->panning.points     = xi.Num_Pan_Points;
      in[i]->panning.sustain    = xi.Pan_Sus;
      in[i]->panning.loop_start = xi.Pan_Loop_Start;
      in[i]->panning.loop_end   = xi.Pan_Loop_End;
      in[i]->panning.type       = xi.Pan_Type;
      in[i]->panning.fadeout    = 0; // no fadeout for pan

      for (int point = 0; point < 24; point++)
	{
	  in[i]->panning.envelope[point] = xi.Pan_Envelope[point];
	}

      // assign samples to keys
      for (int key = 0; key < 96; key++)
	{
	  in[i]->key_samp[key] = xi.Sample_Number[key];
	}

#ifdef DEBUG
      printf("Vibrato type   %d\n",in[i]->vibrato.type);
      printf("Vibrato sweep  %d\n",in[i]->vibrato.sweep);
      printf("Vibrato depth  %d\n",in[i]->vibrato.depth);
      printf("Vibrato rate   %X\n",in[i]->vibrato.rate);
      printf("Volume points  %d\n",in[i]->volume.points);
      printf("Volume sustain %d\n",in[i]->volume.sustain);
      printf("Volume start   %X\n",in[i]->volume.loop_start);
      printf("Volume end     %X\n",in[i]->volume.loop_end);
      printf("Volume Type    %d\n",in[i]->volume.type);
      printf("Volume fadeout %X\n",in[i]->volume.fadeout);
#endif

      ////////////////////
      // SAMPLE HEADERS //
      ////////////////////

      // allocate space for samples
      in[i]->samples = (Sample **)calloc(in[i]->NumSamples,sizeof(Sample *));

      if (!in[i]->samples)
	{
	  error_message = "calloc error";
	  return 0;
	}

      SMPHeader *sheader = 0;
      sheader=(SMPHeader *)malloc(in[i]->NumSamples*sizeof(SMPHeader));

      if (!sheader)
	{
	  error_message = "malloc error";
	  return 0;
	}

      // read in sample headers
      for (int j = 0; j <  in[i]->NumSamples; j++)
	{
#ifdef DEBUG
	  //	  printf("Position %lX\n",ftell(fp));
	  printf("j %d\n",j);
#endif      
	  if (fread(&sheader[j],sizeof(SMPHeader),1,fp)!=1)
	    {
	      perror("fread");
	      error_message = "read error";
	      return 0;
	    }
	  
	  if ((in[i]->samples[j]=(Sample *)calloc(1,sizeof(Sample)))==0)
	    {
	      perror("malloc Sample");
	      dump();
	      error_message = "malloc error";
	      return 0;
	    }

	  // assign sample header values to samples
	  in[i]->samples[j]->length        = sheader[j].Length;
	  in[i]->samples[j]->loop_start    = sheader[j].Loop_Start;

	  in[i]->samples[j]->loop_end      = sheader[j].Loop_Length +
	    sheader[j].Loop_Start-1;	  

	  in[i]->samples[j]->volume        = sheader[j].Volume;
	  in[i]->samples[j]->finetune      = sheader[j].FineTune;
	  in[i]->samples[j]->relative_note = sheader[j].RelativeNote;
	  in[i]->samples[j]->frequency     = 8363; //?

	  if (sheader[j].Type & 0x01)
	    in[i]->samples[j]->flags |= Sample::SMP_LOOP;

	  if (sheader[j].Type & 0x2)
	    { // should I flag SMP_LOOP as well ?
	      in[i]->samples[j]->flags |= Sample::SMP_LOOP; 
	      in[i]->samples[j]->flags |= Sample::SMP_BIDI;	  
	    }

	  if (sheader[j].Type & 0x10) // 16 bit sample
	    {
	      in[i]->samples[j]->flags |= Sample::SMP_16BIT;
#ifdef DEBUG
	      printf("16 bit\n");
#endif
	    }

	  in[i]->samples[j]->panning = sheader[j].Panning;
	  in[i]->samples[j]->name = strdup(sheader[j].Name);
	}

      /////////////////
      // SAMPLE DATA //
      /////////////////

      for (int j = 0; j <  in[i]->NumSamples; j++)
	{

	  if (!sheader[j].Length)
	    {
	      // in[i]->Samples[j] = 0;
	      continue;
	    }

#ifdef DEBUG
	  printf("length %lX\n", sheader[j].Length);
#endif      	  

	  if ((data = (byte *)calloc(sheader[j].Length,sizeof(byte)))==0)
	    {
	      perror("malloc");
	      error_message = "malloc error";
	      return 0;
	    }

	  // read in sample data
	  if (fread(data,sheader[j].Length,1,fp)!=1)
	    {
	      if (ferror(fp))
		{
		  perror("fread");
		  fclose(fp);
		  error_message = "read error";
		  return 0;
		}
	    }

	  // malloc space for Samples[i]->Data
	  if ((in[i]->samples[j]->data =
	       (byte *)calloc(sheader[j].Length,sizeof(byte))) == 0)
	    {
	      perror("malloc");
	      dump();
	      error_message = "malloc error";
	      return 0;
	    }

	  // convert delta values to PCM
	  if (in[i]->samples[j]->flags & Sample::SMP_16BIT) // 16 bit
	    {
	      sword old = 0;
	      sword newp;
	      // convert lengths
	      in[i]->samples[j]->length     >>= 1;
	      in[i]->samples[j]->loop_start >>= 1;
	      in[i]->samples[j]->loop_end   >>= 1;

	      /* convert data to signed 16 bit from deltas */
	      for (udword x=0; x < in[i]->samples[j]->length;x++) 
		{
		  /* first need to construct a word from 2 bytes */
		  newp = *(data + 2*x) + (*(data+2*x+1) << 8);
		  newp += old;
		  /* LOWER byte */
		  *(in[i]->samples[j]->data+2*x) = byte(newp & 0xff); 
		  /* UPPER byte */ 
		  *(in[i]->samples[j]->data+2*x+1) = byte((newp>>8) & 0xff);
		  old = newp;
		}
	    }
	  else // 8 bit
	    {
	      sbyte old = 0;
	      sbyte newp;

	      for (udword x = 0; x < in[i]->samples[j]->length; x++)
		{
		  newp = *(data + x) + old;
		  *(in[i]->samples[j]->data+x) = newp ^ 0x80;
		  old = newp;
		}
	    }
	  free(data);
	  data = 0;
	}    
      free(sheader);
    }
  // copy pointers to Samples from Instuments
  // malloc sample space
  if ((samples = (Sample **)calloc(NumSamples, sizeof(Sample *)))==0)
    {
      perror("malloc SAMPLES *");
      error_message = "malloc error";
      return 0;
    }

  // map instrument samples to samples
  int xi = 0, xs = 0;
  
  for (int s = 0; (s < NumSamples) && (xi < NumInst); s++)
    {
      if (in[xi]->NumSamples > xs)
	{
	  samples[s] = in[xi]->samples[xs];
	  xs++;
	}
      else
	{
	  xi++;
	  xs = 0;
	}
    }
#ifdef DEBUG
  printf("samples %d\n",NumSamples);
#endif      
  return 1;
}

// this is not fully implemented yet
int Module::loadIT(const char *filename)
{ // load it
  IT_HEADER Head;
  dword *para_inst=0;
  dword *para_smp=0;
  dword *para_pat=0;
  FILE *fp = fopen(filename, "rb");
    
  if (!fp) 
    {
      dump();
      error_message = "No Such File";
      return 0;
    }

  if (fread(&Head,sizeof(IT_HEADER),1,fp)!=1)
    {
      error_message = "read error";
      return 0;
    }

  name        = strdup(Head.name);
  // Channels    = Head.Channels;
  NumSamples  = Head.Samples;
  NumPatterns = Head.Patterns;
  SongLength  = Head.SongLength;
  speed       = Head.speed;
  bpm         = Head.bpm;

#ifdef DEBUG
  printf("IMPM           %.4s\n",  Head.IMPM);
  printf("name           %.26s\n", name);
  printf("SongLength     %X\n",    SongLength);
  printf("Instruments    %X\n",    Head.Instruments);
  printf("Samples        %X\n",    Head.Samples);
  printf("Patterns       %X\n",    Head.Patterns);
  printf("cwtv           %X\n",    Head.cwtv);
  printf("cmwt           %X\n",    Head.cmwt);

  //flags
  if (Head.flags&0x01)
    printf("stereo\n");
  else
    printf("mono\n");

  if (Head.flags&0x02) // ignore this bit -- don't mix 0 vol samples anyway
    printf("0 volume optimisations\n");

  if (Head.flags&0x04)
    printf("Instruments on\n");

  if (Head.flags&0x08)
    printf("Linear slides\n");
  else
    printf("Amiga slides\n");

  if (Head.flags&0x10)
    printf("Old effects\n");

  if (Head.flags&0x20)
    printf("Link memory\n");

  printf("Global Volume  %X\n",Head.Global_Volume);
  printf("Master Volume  %X\n",Head.Master_Volume);
  printf("speed          %d\n",speed);
  printf("bpm            %d\n",bpm);
  printf("Separation     %X\n",Head.Separation);

  if (Head.special) 
    {
      printf("Message Length %X\n",Head.msg_length);
      printf("Message Offset %lX\n",Head.msg_offset);
    }

#endif
  if ((order = (byte *)malloc(SongLength))==0)
    {
      error_message = "malloc error";
      return 0;
    }
  if (fread(order, SongLength,1,fp)!=1)
    {
      error_message = "read error: order";
      return 0;
    }

  // need to remap the order to remove marker patterns -- 254,255
  int real_songlength = 0;
  for (int i = 0; i < SongLength; i++)
    {
      if (order[i] < 254) 
	{
	  if (order[i] > NumPatterns)
	    NumPatterns = order[i];
	  order[real_songlength++] = order[i];	  
	}
    }
  NumPatterns++;
  SongLength = real_songlength;
#ifdef DEBUG
  printf("Biggest Pattern %X\n", NumPatterns);

  for (int i = 0; i < SongLength; i++)
    printf("[%.2X : %.2X] ", i, order[i]);

  printf("\n");
#endif
  // parapointers -- similar to s3m format
  // Instruments  -- NumInst          * dword 
  // Samples      -- NumSamples       * dword
  // Patterns     -- Head.NumPatterns * dword
  if ((para_inst = (dword *)malloc(Head.Instruments*4))==0)
    {
      error_message = "malloc error";
      return 0;
    }

  if (fread(para_inst,Head.Instruments*4,1,fp)!=1)
    {
      error_message = "read error: para inst";
      return 0;
    }

#ifdef DEBUG
  for (int i = 0; i < Head.Instruments; i++)
    printf("%lX ", para_inst[i]);

  printf("\n");
#endif
  if ((para_smp = (dword *)malloc(Head.Samples*4))==0)
    {
      error_message = "malloc error";
      return 0;
    }

  if (fread(para_smp, Head.Samples * 4, 1, fp)!=1)
    {
      error_message = "read error: para inst";
      return 0;
    }

#ifdef DEBUG
  for (int i = 0; i < Head.Samples; i++)
    printf("%lX ", para_smp[i]);

  printf("\n");
#endif

  if ((para_pat = (dword *)malloc(Head.Patterns*4))==0)
    {
      error_message = "malloc error";
      return 0;
    }

  if (fread(para_pat,Head.Patterns*4,1,fp)!=1)
    {
      error_message = "read error: para inst";
      return 0;
    }

#ifdef DEBUG
  for (int i = 0; i < Head.Patterns; i++)
    printf("%lX ", para_pat[i]);

  printf("\n");
#endif
  // okay we've got the para pointers, now use them
  // malloc mod->Samples space
  if ((samples=(Sample **)calloc(Head.Samples,sizeof(Sample *)))==0) 
    {
      perror("malloc SAMPLES *");
      error_message = "malloc error";
      return 0;
    }

  // read in Samples
  for (int i = 0; i < Head.Samples; i++)
    { // for
      if (para_smp[i])
	{
	  if (!fseek(fp,para_smp[i],SEEK_SET))
	    {
	      IT_SMP smp;
	      if (fread(&smp,sizeof(IT_SMP),1,fp)!=1)
		{
		  printf("\nerror\n%X %lX\n", i, para_smp[i]);
		  dump();
		  error_message = "read error";
		  return 0;
		}

	      if ((samples[i]=(Sample *)calloc(1,sizeof(Sample)))==0) 
		{
		  perror("malloc Sample");
		  dump();
		  error_message = "malloc error";
		  return 0;
		}
	      samples[i]->length = smp.length;

	      if (smp.flags & 0x10)
		{
		  samples[i]->flags |=Sample::SMP_LOOP;
		  samples[i]->loop_start = smp.loop_start;
		  samples[i]->loop_end = smp.loop_end;
		}

	      if (smp.flags & 0x02) // 16 bit sample
		samples[i]->flags |=Sample::SMP_16BIT;	      

	      if (smp.flags & 0x04) // stereo
		samples[i]->flags |=Sample::SMP_STEREO;

	      samples[i]->volume = smp.Volume;
	      samples[i]->finetune = 0;
	      samples[i]->frequency = smp.C5Speed;
	      samples[i]->name = strdup(smp.name);

	      // malloc space for samples[i]->Data
	      if((samples[i]->data=(byte *)malloc(smp.length))==0)
		{
		  perror("malloc");
		  dump();
		  error_message = "malloc error";
		  return 0;
		}

	      if (smp.offset)
		if (!fseek(fp,smp.offset,SEEK_SET))
		  {
		    if (fread(samples[i]->data,smp.length,1,fp)!=1) 
		      {
			dump();
			error_message = "read error";
			return 0;
		      }

		    if (smp.cvt & 0x1) //samples are signed
		      {
#ifdef DEBUG
			printf("signed");
#endif
			// convert to unsigned			
			if (samples[i]->flags & Sample::~SMP_16BIT) 
			  for (unsigned int j = 0; 
			       j < samples[i]->length; j++) 
			    *(samples[i]->data + j) ^= 0x80;
		      }
		  }
#ifdef DEBUG
	      if (!strncmp((char *)smp.id, "IMPS", 4))
		{
		  printf("%d================%d\n",i,i);
		  if (smp.name && *smp.name)
		    printf("name           [%.26s]\n",smp.name);
		  if (smp.filename && *smp.filename)
		    printf("filename       [%.12s]\n",smp.filename);
		  if (smp.Global_Volume != 64)
		    printf("Global Volume  %d\n",smp.Global_Volume);
#if 0
		  if (smp.flags&0x01)
		    printf("sample associated\n");
		  if (smp.flags&0x02)
		    printf("16 bit\n");
		  if (smp.flags&0x04)
		    printf("stereo\n");
		  if (smp.flags&0x10)
		    printf("looped\n");		
		  if (smp.flags&0x20)
		    printf("sustain loop\n");
		  if (smp.flags&0x40)
		    printf("ping-pong loop\n");
		  if (smp.flags&0x80)
		    printf("ping-pong sustain loop\n");
		  printf("cvt            %d\n",smp.cvt); // convert ?
		  printf("Default Pan    %d\n",smp.Default_Pan);
#endif
		  if (smp.Volume != 64)
		    printf("Volume         %d\n",smp.Volume);
		  if (samples[i]->Length)
		    {
		      printf("Length         %lX\n", samples[i]->Length);
		      if (smp.offset)
			printf("offset %lX\n",smp.offset);
		    }
		  if (smp.flags&0x10)
		    {
		      printf("loop start     %lX\n",smp.loop_start);
		      printf("loop end       %lX\n",smp.loop_end);
		    }

		  if (smp.C5Speed!=8363)
		    printf("C5Speed        %ld\n",smp.C5Speed);

#ifdef 0
		  printf("SusLoopStart   %lX\n",smp.SusLoopStart);
		  printf("SusLoopEnd     %lX\n",smp.SusLoopEnd);
		  printf("Vibrato Speed  %d\n",smp.Vibrato_Speed);
		  printf("Vibrato Depth  %d\n",smp.Vibrato_Depth);
		  printf("Vibrato Rate   %d\n",smp.Vibrato_Rate);
		  printf("Vibrato Type   %d\n",smp.Vibrato_Type);
#endif
		}
#endif	      
	    }
	  else
	    {
	      error_message = "seek error";
	      return 0;
	    }
	}
    }

  // allocate space for instruments
  in=(Instrument**)calloc(NumInst,sizeof(Instrument *));

  if (!in)
    {
      error_message = "calloc error";
      return 0;
    }

  // read in Instruments
#ifdef DEBUG
  printf("==INSTRUMENTS==\n");
#endif
  for (int i = 0; i < Head.Instruments; i++)
    { 
      if (para_inst[i])
	{ 
	  if (!fseek(fp,para_inst[i], SEEK_SET))
	    { 
	      IT_INST it_in;
	      if (fread(&it_in, 547,1,fp)!=1)
		{
		  printf("\nerror inst\n%X %lX\n", i, para_inst[i]);
		  error_message = "read error";
		  return 0;
		}
	      // assign values to instruments 
	      // IT associates vibrato with samples, 
	      // need to change data structure to accomadate this

	      // volume envelope
	      in[i]->volume.points     = it_in.Volume.points;
	      in[i]->volume.sustain    = it_in.Volume.sustain_start;
	      in[i]->volume.loop_start = it_in.Volume.loop_start;
	      in[i]->volume.loop_end   = it_in.Volume.loop_end;
	      //in[i]->Volume.Type       = it_in.Volume.
	      // no fade out for IT envelopes
	      // IT has up to 25 sets !
#if 0
	      for (int point = 0; point < 24; point++)
		in[i]->volume.Envelope[point] = xi.Volume._Envelope[point];
	      // panning	  
	      in[i]->Panning.Points     = xi.Num_Pan_Points;
	      in[i]->Panning.Sustain    = xi.Pan_Sus;
	      in[i]->Panning.Loop_Start = xi.Pan_Loop_Start;
	      in[i]->Panning.Loop_End   = xi.Pan_Loop_End;
	      in[i]->Panning.Type       = xi.Pan_Type;
	      in[i]->Panning.Fadeout    = 0; // no fadeout for pan
	      for (int point = 0; point < 24; point++)
		in[i]->Panning.Envelope[point]=xi.Pan_Envelope[point];
	      for (int key = 0; key < 96; key++)
		in[i]->key_samp[key] = xi.Sample_Number[key];
#endif
#ifdef DEBUG
	      if (!strncmp((char *)it_in.id, "IMPI",4))
		{
		  printf("name [%.26s]\n",it_in.name);
		  if (it_in.filename)
		    if (*it_in.filename)
		      printf("filename [%.12s]\n",it_in.filename);
		  if (it_in.NNA)
		    printf("NNA %d\n",it_in.NNA);
		  if (it_in.DCT)
		    printf("DCT %d\n",it_in.DCT);
		  if (it_in.DCA)
		    printf("DCA %d\n",it_in.DCA);
		  if (it_in.fadeout)
		    printf("fadeout %d\n",it_in.fadeout);
		  if (it_in.PPS)
		    printf("PPS %d\n",it_in.PPS);
		  if (it_in.PPC != 60)
		    printf("PPC %d\n",it_in.PPC);
		  if (it_in.Global_Volume != 128)
		    printf("Global_Volume %d\n",it_in.Global_Volume);
		  if (it_in.Default_Pan != 160)
		    printf("Default Pan %d\n",it_in.Default_Pan);
		  if (it_in.RV)
		    printf("RV %d\n",it_in.RV);
		  if (it_in.RP)
		    printf("RP %d\n",it_in.RP);
		  if (it_in.version)
		    printf("version %d\n",it_in.version);
		  if (it_in.Samples)
		    printf("Samples %d\n",it_in.Samples);
		  if (it_in.midi_channel)
		    printf("midi_channel %d\n",it_in.midi_channel);
		  if (it_in.midi_program)
		    printf("midi_program %d\n",it_in.midi_program);
		  if (it_in.midi_bank && it_in.midi_bank !=0xffff)
		    printf("midi_bank %d\n",it_in.midi_bank);
		}
#endif
	    }
	}
    }
  // read in Patterns
  //  for (int p = 0; p < Head.Patterns; p++)

  fclose(fp);
  error_message = "\nNOT FINISHED YET!!!\n";
  return 0;
}

